<?php
/**
 * TidePoolUI REST API Entry Point
 * Using Enchilada REST Library
 */

require_once __DIR__ . '/../includes/bootstrap.inc.php';
require_once APPLICATION_LIBDIR . 'REST/EnchiladaREST.class.php';
require_once APPLICATION_LIBDIR . 'REST/EnchiladaRequest.class.php';
require_once APPLICATION_LIBDIR . 'REST/EnchiladaResponse.class.php';
require_once APPLICATION_CLASSDIR . 'geoip.class.php';

// Create API router
$api = new EnchiladaREST('/api/v1');

// Enable CORS
$api->enableCors([
    'origins' => $SETTINGS['api']['cors_origins'] ?? '*',
]);

// Health check
$api->get('/health', function($req, $res) {
    $res->success([
        'status' => 'ok',
        'version' => APPLICATION_VERSION,
        'timestamp' => date('c'),
    ]);
});

// Helper: Check if request has valid admin secret
function isAdminRequest($req, $settings): bool {
    $secret = $settings['admin']['secret'] ?? null;
    if (empty($secret)) {
        return false; // No secret configured = no admin access
    }
    $provided = $req->getHeader('X-Admin-Secret');
    return $provided === $secret;
}

// Helper: Mask IP addresses in data array
function maskIPsInData(array $data, string $field = 'address'): array {
    $geoip = GeoIP::getInstance();
    foreach ($data as &$item) {
        if (isset($item[$field])) {
            $item['location'] = $geoip->maskIP($item[$field]);
            unset($item[$field]); // Remove raw IP
        }
    }
    return $data;
}

// Get app config (for frontend feature flags)
$api->get('/config', function($req, $res) use ($SETTINGS) {
    $res->json([
        'demo_enabled' => !empty($SETTINGS['app']['demo_enabled']),
        'admin_enabled' => !empty($SETTINGS['admin']['secret']),
    ]);
});

// Verify admin secret
$api->post('/admin/verify', function($req, $res) use ($SETTINGS) {
    if (isAdminRequest($req, $SETTINGS)) {
        $res->success(['admin' => true]);
    } else {
        $res->error('Invalid secret', 401);
    }
});

// Get recent shares
$api->get('/shares', function($req, $res) use ($SETTINGS) {
    $limit = min(100, max(1, intval($req->getQueryParam('limit', 50))));
    $poolId = $req->getQueryParam('pool');
    if ($poolId === '' || $poolId === 'all') {
        $poolId = null;
    }
    
    $store = ShareStore::getInstance();
    $shares = $store->getRecentShares($limit, $poolId);
    
    // Mask IPs for non-admin users
    if (!isAdminRequest($req, $SETTINGS)) {
        $shares = maskIPsInData($shares);
    }
    
    // Return {shares: [...]} for Handlebars {{#each shares}}
    $res->json(['shares' => $shares]);
});

// Get pool statistics (supports ?pool=<pool_id> filter)
$api->get('/stats', function($req, $res) {
    $poolId = $req->getQueryParam('pool');
    // Treat empty string or 'all' as null (aggregate)
    if ($poolId === '' || $poolId === 'all') {
        $poolId = null;
    }
    
    $store = ShareStore::getInstance();
    $stats = $store->getPoolStats($poolId);
    $hashrate = $store->getPoolHashrate($poolId);
    // Merge stats with hashrate data
    $res->json(array_merge($stats, $hashrate));
});

// Get list of known pools
$api->get('/pools', function($req, $res) {
    $store = ShareStore::getInstance();
    $pools = $store->getPools();
    // Ensure we return an array, not a Redis set object
    $res->json(['pools' => array_values($pools)]);
});

// Get worker statistics
$api->get('/workers', function($req, $res) use ($SETTINGS) {
    $poolId = $req->getQueryParam('pool');
    if ($poolId === '' || $poolId === 'all') {
        $poolId = null;
    }
    
    $store = ShareStore::getInstance();
    $workers = $store->getWorkerStats($poolId);
    
    // Add hashrate to each worker
    foreach ($workers as &$worker) {
        $workerName = $worker['workername'] ?? '';
        if ($workerName) {
            $hashrate = $store->getWorkerHashrate($workerName);
            $worker = array_merge($worker, $hashrate);
        }
    }
    
    // Mask IPs for non-admin users
    if (!isAdminRequest($req, $SETTINGS)) {
        $workers = maskIPsInData($workers);
    }
    
    // Return {workers: [...]} for Handlebars {{#each workers}}
    $res->json(['workers' => $workers]);
});

// Add demo data (for testing without Kafka)
$api->post('/demo', function($req, $res) use ($SETTINGS) {
    // Check if demo mode is enabled
    if (empty($SETTINGS['app']['demo_enabled'])) {
        $res->error('Demo mode is disabled', 403);
        return;
    }
    
    $store = ShareStore::getInstance();
    $dateFormat = $SETTINGS['date']['format'] ?? 'c';
    $workers = ['worker1.rig1', 'worker1.rig2', 'miner2.antminer', 'pool.test'];
    // Mix of private and public IPs for testing GeoIP
    $addresses = [
        '192.168.1.100',  // Private
        '10.0.0.50',      // Private
        '8.8.8.8',        // Google DNS (Mountain View, CA)
        '1.1.1.1',        // Cloudflare (San Francisco, CA)
        '208.67.222.222', // OpenDNS (San Francisco, CA)
    ];
    $now = time();
    
    for ($i = 0; $i < 10; $i++) {
        $worker = $workers[array_rand($workers)];
        $username = explode('.', $worker)[0];
        $valid = rand(0, 100) > 5; // 95% valid
        
        $store->addShare([
            'workinfoid' => rand(100000, 999999),
            'clientid' => rand(1, 100),
            'enonce1' => bin2hex(random_bytes(4)),
            'nonce2' => bin2hex(random_bytes(8)),
            'nonce' => bin2hex(random_bytes(4)),
            'ntime' => dechex($now - rand(0, 60)),
            'diff' => 1.0,
            'sdiff' => round(rand(100, 5000) / 1000, 3),
            'hash' => '0000000000000' . bin2hex(random_bytes(20)),
            'result' => $valid,
            'errn' => $valid ? 0 : 21,
            'createdate' => date($dateFormat, $now - rand(0, 60)),
            'workername' => $worker,
            'username' => $username,
            'address' => $addresses[array_rand($addresses)],
            'agent' => 'cgminer/4.11.1',
        ]);
    }
    
    $res->success(['message' => 'Added 10 demo shares']);
});

// Run the API
$api->run();
