#!/bin/sh
# TidePoolUI Development Helper Script
# Usage: ./dev.sh [command]
#
# On FreeBSD: Install PHP with: pkg install php83
# Then run: ./dev.sh start

set -e

PROJECT_DIR="$(cd "$(dirname "$0")" && pwd)"

# Find PHP binary (try common names)
find_php() {
    for cmd in php83 php82 php81 php; do
        if command -v "$cmd" >/dev/null 2>&1; then
            echo "$cmd"
            return 0
        fi
    done
    return 1
}

case "${1:-help}" in
    start)
        echo "Starting TidePoolUI development server..."
        
        PHP_BIN=$(find_php) || {
            echo "Error: PHP not found."
            echo ""
            echo "Install PHP on FreeBSD with:"
            echo "  sudo pkg install php83"
            echo ""
            exit 1
        }
        
        # Get server IP address
        SERVER_IP=$(ifconfig | grep -E 'inet [0-9]' | grep -v '127.0.0.1' | head -1 | awk '{print $2}')
        
        echo "Using: $PHP_BIN"
        echo ""
        echo "TidePoolUI is running at: http://${SERVER_IP}:8080"
        echo "API endpoint: http://${SERVER_IP}:8080/api/v1/"
        echo ""
        echo "Press Ctrl+C to stop"
        echo ""
        
        cd "$PROJECT_DIR/frontend"
        exec "$PHP_BIN" -S 0.0.0.0:8080 router.php
        ;;
    
    stop)
        echo "To stop the server, press Ctrl+C in the terminal where it's running."
        ;;
    
    install)
        echo "Installing PHP 8.3 on FreeBSD..."
        sudo pkg install -y php83
        echo ""
        echo "PHP installed. Now run: ./dev.sh start"
        ;;
    
    help|*)
        echo "TidePoolUI Development Script"
        echo ""
        echo "Usage: $0 [command]"
        echo ""
        echo "Commands:"
        echo "  start    Start the PHP development server on 0.0.0.0:8080"
        echo "  stop     Instructions to stop the server"
        echo "  install  Install PHP 8.3 via pkg (FreeBSD)"
        echo "  help     Show this help message"
        echo ""
        echo "Prerequisites:"
        echo "  FreeBSD: pkg install php83"
        ;;
esac
