/**
 * TidePool Monitor - Application JavaScript
 * 
 * Handlebars helpers are in helpers.js (loaded before this file)
 */

// Admin mode management
const AdminMode = {
    STORAGE_KEY: 'tidepoolui_admin_secret',
    
    getSecret: function() {
        return localStorage.getItem(this.STORAGE_KEY);
    },
    
    setSecret: function(secret) {
        if (secret) {
            localStorage.setItem(this.STORAGE_KEY, secret);
        } else {
            localStorage.removeItem(this.STORAGE_KEY);
        }
        this.updateUI();
        this.refreshData();
    },
    
    isEnabled: function() {
        return !!this.getSecret();
    },
    
    updateUI: function() {
        const btn = document.getElementById('admin-toggle');
        if (btn) {
            btn.textContent = this.isEnabled() ? '🔓' : '🔒';
            btn.title = this.isEnabled() ? 'Admin mode (click to lock)' : 'Click to unlock admin mode';
        }
    },
    
    refreshData: function() {
        // Trigger HTMX to reload data with new headers
        const workers = document.getElementById('workers-container');
        const shares = document.getElementById('shares-container');
        if (workers) htmx.trigger(workers, 'htmx:load');
        if (shares) htmx.trigger(shares, 'htmx:load');
    },
    
    promptForSecret: async function() {
        const secret = prompt('Enter admin secret:');
        if (!secret) return;
        
        // Verify the secret with the API
        try {
            const response = await fetch('/api/v1/admin/verify', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Admin-Secret': secret
                }
            });
            
            if (response.ok) {
                this.setSecret(secret);
                alert('Admin mode enabled');
            } else {
                alert('Invalid secret');
            }
        } catch (e) {
            alert('Failed to verify secret');
        }
    },
    
    toggle: function() {
        if (this.isEnabled()) {
            this.setSecret(null);
        } else {
            this.promptForSecret();
        }
    }
};

// Configure HTMX to send admin secret header
document.addEventListener('htmx:configRequest', function(evt) {
    const secret = AdminMode.getSecret();
    if (secret) {
        evt.detail.headers['X-Admin-Secret'] = secret;
    }
});

// Theme toggle
document.addEventListener('DOMContentLoaded', function() {
    const themeToggle = document.getElementById('theme-toggle');
    const html = document.documentElement;
    
    // Check for saved theme preference
    const savedTheme = localStorage.getItem('theme') || 'dark';
    html.setAttribute('data-theme', savedTheme);
    updateThemeIcon(savedTheme);
    
    themeToggle.addEventListener('click', function(e) {
        e.preventDefault();
        const currentTheme = html.getAttribute('data-theme');
        const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
        
        html.setAttribute('data-theme', newTheme);
        localStorage.setItem('theme', newTheme);
        updateThemeIcon(newTheme);
    });
    
    function updateThemeIcon(theme) {
        themeToggle.textContent = theme === 'dark' ? '☀️' : '🌙';
    }
    
    // Clock
    function updateClock() {
        const clock = document.getElementById('clock');
        if (clock) {
            clock.textContent = new Date().toLocaleTimeString();
        }
    }
    updateClock();
    setInterval(updateClock, 1000);
    
    // Admin toggle button
    const adminToggle = document.getElementById('admin-toggle');
    if (adminToggle) {
        AdminMode.updateUI();
        adminToggle.addEventListener('click', function(e) {
            e.preventDefault();
            AdminMode.toggle();
        });
    }
});

// Console welcome message
console.log('%c🌊 TidePool Monitor', 'font-size: 24px; font-weight: bold; color: #0ea5e9;');
console.log('Powered by SeaTidePool + HTMX + Handlebars');
