<?php
/**
 * GeoIP - IP address geolocation and privacy utilities
 * 
 * Provides GeoIP lookup with caching and private IP detection.
 * Uses ip-api.com free tier (limited to 45 requests/minute).
 */
class GeoIP
{
    private static ?GeoIP $instance = null;
    
    private ?Redis $redis = null;
    private int $cacheTtl = 86400; // 24 hours
    
    private const CACHE_PREFIX = 'tidepoolui:geoip:';
    private const API_URL = 'http://ip-api.com/json/';
    
    private function __construct()
    {
        try {
            $this->redis = new Redis();
            $host = getenv('REDIS_HOST') ?: '127.0.0.1';
            $port = (int)(getenv('REDIS_PORT') ?: 6379);
            $this->redis->connect($host, $port);
        } catch (Exception $e) {
            // Redis not available, will work without cache
            $this->redis = null;
        }
    }
    
    public static function getInstance(): GeoIP
    {
        if (self::$instance === null) {
            self::$instance = new GeoIP();
        }
        return self::$instance;
    }
    
    /**
     * Check if an IP address is private/non-routable
     */
    public function isPrivateIP(string $ip): bool
    {
        // Handle empty or invalid
        if (empty($ip)) {
            return true;
        }
        
        // filter_var with FILTER_FLAG_NO_PRIV_RANGE and FILTER_FLAG_NO_RES_RANGE
        // returns false for private/reserved IPs
        $result = filter_var(
            $ip,
            FILTER_VALIDATE_IP,
            FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE
        );
        
        // If result is false, the IP is private/reserved (or invalid)
        return $result === false;
    }
    
    /**
     * Get location string for an IP address
     * Returns "City, State, Country" format, omitting unknown parts
     */
    public function getLocation(string $ip): ?string
    {
        // Don't lookup private IPs
        if ($this->isPrivateIP($ip)) {
            return null;
        }
        
        // Check cache first
        $cached = $this->getFromCache($ip);
        if ($cached !== null) {
            return $cached;
        }
        
        // Lookup from API
        $location = $this->lookupIP($ip);
        
        // Cache the result (even if null, to avoid repeated lookups)
        $this->saveToCache($ip, $location ?? '');
        
        return $location;
    }
    
    /**
     * Mask an IP address with its location (or return as-is for private IPs)
     */
    public function maskIP(string $ip): string
    {
        if ($this->isPrivateIP($ip)) {
            return $ip; // Return private IPs as-is
        }
        
        $location = $this->getLocation($ip);
        return $location ?: 'Unknown Location';
    }
    
    /**
     * Lookup IP using ip-api.com
     */
    private function lookupIP(string $ip): ?string
    {
        $url = self::API_URL . urlencode($ip) . '?fields=status,city,regionName,country';
        
        $context = stream_context_create([
            'http' => [
                'timeout' => 2, // 2 second timeout
                'ignore_errors' => true,
            ],
        ]);
        
        $response = @file_get_contents($url, false, $context);
        if ($response === false) {
            return null;
        }
        
        $data = json_decode($response, true);
        if (!$data || ($data['status'] ?? '') !== 'success') {
            return null;
        }
        
        // Build location string: City, State, Country
        $parts = [];
        if (!empty($data['city'])) {
            $parts[] = $data['city'];
        }
        if (!empty($data['regionName'])) {
            $parts[] = $data['regionName'];
        }
        if (!empty($data['country'])) {
            $parts[] = $data['country'];
        }
        
        return !empty($parts) ? implode(', ', $parts) : null;
    }
    
    /**
     * Get cached location for IP
     */
    private function getFromCache(string $ip): ?string
    {
        if (!$this->redis) {
            return null;
        }
        
        $key = self::CACHE_PREFIX . $ip;
        $cached = $this->redis->get($key);
        
        if ($cached === false) {
            return null; // Not in cache
        }
        
        return $cached === '' ? null : $cached;
    }
    
    /**
     * Save location to cache
     */
    private function saveToCache(string $ip, string $location): void
    {
        if (!$this->redis) {
            return;
        }
        
        $key = self::CACHE_PREFIX . $ip;
        $this->redis->setex($key, $this->cacheTtl, $location);
    }
}
