# Local Development Setup

This guide covers setting up TidePoolUI for local development on FreeBSD.

## Prerequisites

- FreeBSD 14.x or later
- PHP 8.3 with extensions: redis, rdkafka, json
- Redis server (or Valkey)
- Access to a Kafka cluster
- SeaTidePool running with Kafka enabled

## Quick Start

### 1. Install Dependencies

```sh
sudo pkg install -y php83 php83-extensions php83-pecl-redis php83-pecl-rdkafka redis
```

Enable the extensions if not already:
```sh
sudo cp /usr/local/etc/php/ext-20-redis.ini.sample /usr/local/etc/php/ext-20-redis.ini
sudo cp /usr/local/etc/php/ext-20-rdkafka.ini.sample /usr/local/etc/php/ext-20-rdkafka.ini
```

### 2. Start Redis

```sh
sudo service redis enable
sudo service redis start
```

Verify:
```sh
redis-cli ping
# Should return: PONG
```

### 3. Start the Development Server

```sh
./dev.sh start
```

Or manually:
```sh
php -S 0.0.0.0:8080 -t frontend backend/system/router.php
```

The UI is now available at http://localhost:8080

### 4. Start the Kafka Consumer

In a separate terminal:
```sh
./backend/bin/share-consumer.php
```

You should see:
```
TidePoolUI Kafka Share Consumer
================================
Brokers: localhost:9092
Topic: tidepool.dev.shares
Group: tidepoolui-consumer

Subscribed to tidepool.dev.shares. Waiting for messages...
```

## Configuration

### Redis Connection

Edit `backend/config/settings.php`:

```php
return [
    'redis' => [
        'host' => '127.0.0.1',
        'port' => 6379,
        'prefix' => 'tidepoolui:',
    ],
    // ...
];
```

### Kafka Connection

The consumer reads from environment variables or uses defaults:

```sh
export KAFKA_BROKERS="localhost:9092"
export KAFKA_TOPIC="tidepool.dev.shares"
export KAFKA_GROUP="tidepoolui-consumer"
./backend/bin/share-consumer.php
```

## Testing with Demo Data

If you don't have SeaTidePool running, use demo data:

```sh
curl -X POST http://localhost:8080/api/v1/demo
```

Or click "Add Demo Data" in the UI.

## Verifying the Pipeline

### Check Kafka Messages

```sh
sudo podman exec kafka /usr/local/share/java/kafka/bin/kafka-console-consumer.sh \
    --bootstrap-server localhost:9092 \
    --topic tidepool.dev.shares \
    --from-beginning \
    --max-messages 5
```

### Check Redis Data

```sh
# Count shares
redis-cli llen tidepoolui:shares

# List workers
redis-cli smembers tidepoolui:workers

# Get pool stats
redis-cli hgetall tidepoolui:pool_stats
```

### Check API

```sh
curl -s http://localhost:8080/api/v1/stats | jq
curl -s http://localhost:8080/api/v1/workers | jq
curl -s http://localhost:8080/api/v1/shares?limit=5 | jq
```

## Clearing Data

To reset all data:

```sh
redis-cli del tidepoolui:shares tidepoolui:workers tidepoolui:pool_stats
redis-cli keys "tidepoolui:worker:*" | xargs -r redis-cli del
```

Or via API:
```sh
curl -X DELETE http://localhost:8080/api/v1/data
```

## Container Testing with FreeBSD OCI

For isolated testing, use FreeBSD-native OCI containers (not Linux):

### Prerequisites

```sh
# Install podman suite
pkg install podman-suite

# Configure pf firewall for container networking
cp /usr/local/etc/containers/pf.conf.sample /etc/pf.conf
# Edit /etc/pf.conf to set your network interface (e.g., vmx0, em0)
sysctl net.pf.filter_local=1
sysrc pf_enable=YES
service pf start
```

### Build Test Container

```sh
# Create a test Containerfile
cat > /tmp/Containerfile.tidepoolui << 'EOF'
FROM ghcr.io/freebsd/freebsd-runtime:14.3

RUN pkg install -y apache24 mod_php83 php83 php83-filter php83-pecl-redis

# Copy pre-built packages (from ports)
COPY packages/*.pkg /tmp/
RUN pkg install -y /tmp/*.pkg

EXPOSE 80
CMD ["/usr/local/sbin/httpd", "-D", "FOREGROUND"]
EOF

# Build the image
podman build -t tidepoolui-test:dev -f /tmp/Containerfile.tidepoolui .
```

### Production Deployment

For production, use FreeBSD ports instead of containers:

```sh
# From Morante ports tree
cd /usr/ports-morante/www/tidepoolui-frontend && make install clean
cd /usr/ports-morante/www/tidepoolui-api && make install clean
cd /usr/ports-morante/www/tidepoolui-consumer && make install clean
```

See [RUNBOOK_FREEBSD_INSTALL.md](RUNBOOK_FREEBSD_INSTALL.md) for full deployment guide.

## Troubleshooting

### "Class 'Redis' not found"

Install and enable the PHP Redis extension:
```sh
sudo pkg install php83-pecl-redis
sudo cp /usr/local/etc/php/ext-20-redis.ini.sample /usr/local/etc/php/ext-20-redis.ini
```

### "Class 'RdKafka\Conf' not found"

Install and enable the PHP rdkafka extension:
```sh
sudo pkg install php83-pecl-rdkafka
sudo cp /usr/local/etc/php/ext-20-rdkafka.ini.sample /usr/local/etc/php/ext-20-rdkafka.ini
```

### Consumer Not Receiving Messages

1. Verify Kafka is running: `sudo podman ps | grep kafka`
2. Check topic exists: `kafka-topics.sh --list --bootstrap-server localhost:9092`
3. Verify SeaTidePool has Kafka enabled in its config
4. Check consumer group offset: may need `--from-beginning` for testing

### UI Shows "Loading..." Forever

1. Check browser console for JavaScript errors
2. Verify API is responding: `curl http://localhost:8080/api/v1/health`
3. Check PHP error log for backend issues

## Related Documentation

- [ARCHITECTURE.md](ARCHITECTURE.md) - System architecture
- [API.md](API.md) - API reference
- [SeaTidePool LOCAL_DEVELOPMENT.md](../../SeaTidePool/doc/LOCAL_DEVELOPMENT.md) - Pool setup
