#!/usr/bin/env php
<?php
/**
 * TidePoolBridge - ZMQ to Redis Bridge Daemon
 * 
 * Subscribes to two ZMQ PUB sockets:
 *   1. bitcoind hashblock — triggers JSON-RPC poll for network stats → Redis
 *   2. SeaTidePool poolstats — pool stats JSON → Redis
 * 
 * @author Daniel Morante
 * @copyright 2026 TidePool Project
 * @license BSD-3-Clause
 */

declare(ticks=1);

// --- Bootstrap ---

define('APP_ROOT', dirname(__DIR__) . DIRECTORY_SEPARATOR);
define('APP_NAME', 'TidePoolBridge');
define('APP_VERSION', '0.1.0');

// Autoload Enchilada libraries
spl_autoload_register(function ($class) {
    $prefix = 'Enchilada\\';
    if (strncmp($class, $prefix, strlen($prefix)) !== 0) {
        return;
    }
    $relative = substr($class, strlen($prefix));
    // Map trait files: DaemonBehavior → DaemonBehavior.trait.php
    $file = APP_ROOT . 'libraries/Enchilada/' . str_replace('\\', '/', $relative) . '.php';
    if (file_exists($file)) {
        require_once $file;
        return;
    }
    $traitFile = APP_ROOT . 'libraries/Enchilada/' . str_replace('\\', '/', $relative) . '.trait.php';
    if (file_exists($traitFile)) {
        require_once $traitFile;
    }
});

use Enchilada\Config\IniConfig;

// --- Configuration ---

$configFile = getenv('TIDEPOOLBRIDGE_CONF') ?: APP_ROOT . 'config/settings.ini';
if (!file_exists($configFile)) {
    fprintf(STDERR, "Configuration file not found: %s\n", $configFile);
    fprintf(STDERR, "Copy config/settings.ini.sample to config/settings.ini and edit.\n");
    exit(1);
}

$settings = IniConfig::load($configFile);

// --- Bridge Daemon ---

class ZmqBridge
{
    use \Enchilada\Daemon\DaemonBehavior;
    
    private IniConfig $settings;
    
    // ZMQ config
    private string $bitcoindZmqEndpoint;
    private string $bitcoindRpcUrl;
    private string $bitcoindRpcUser;
    private string $bitcoindRpcPass;
    private string $seatidepoolZmqEndpoint;
    
    // Redis config
    private string $redisHost;
    private int $redisPort;
    private int $redisDb;
    private string $chainId;
    
    // Runtime
    private ?\Redis $redis = null;
    private ?\ZMQPoll $poll = null;
    private ?\ZMQSocket $bitcoindSub = null;
    private ?\ZMQSocket $poolstatsSub = null;
    
    public function __construct(IniConfig $settings)
    {
        $this->settings = $settings;
        
        // bitcoind settings
        $this->bitcoindZmqEndpoint = $settings->reqString('bitcoind', 'zmq_endpoint');
        $this->bitcoindRpcUrl = $settings->reqString('bitcoind', 'rpc_url');
        $this->bitcoindRpcUser = $settings->reqString('bitcoind', 'rpc_user');
        $this->bitcoindRpcPass = $settings->reqString('bitcoind', 'rpc_pass');
        
        // SeaTidePool settings
        $this->seatidepoolZmqEndpoint = $settings->reqString('seatidepool', 'zmq_endpoint');
        
        // Redis settings
        $this->redisHost = $settings->getString('redis', 'host', '127.0.0.1');
        $this->redisPort = $settings->getInt('redis', 'port', 6379);
        $this->redisDb = $settings->getInt('redis', 'db', 0);
        
        // Bridge settings
        $this->chainId = $settings->getString('bridge', 'chain_id', 'bitcoin');
        
        // Initialize daemon behavior
        $this->initDaemon([
            'health_file' => getenv('HEALTH_FILE') ?: '/tmp/tidepoolbridge.health',
            'max_memory_mb' => $settings->getInt('bridge', 'max_memory_mb', 64),
        ]);
        
        // Bridge-specific metrics
        $this->metrics['hashblock_received'] = 0;
        $this->metrics['poolstats_received'] = 0;
        $this->metrics['rpc_calls'] = 0;
        $this->metrics['rpc_errors'] = 0;
        $this->metrics['redis_writes'] = 0;
        $this->metrics['redis_errors'] = 0;
    }
    
    /**
     * Custom health data
     */
    protected function getCustomHealthData(): array
    {
        return [
            'chain_id' => $this->chainId,
            'zmq' => [
                'bitcoind' => $this->bitcoindZmqEndpoint,
                'seatidepool' => $this->seatidepoolZmqEndpoint,
            ],
            'redis' => [
                'host' => $this->redisHost,
                'port' => $this->redisPort,
                'db' => $this->redisDb,
            ],
        ];
    }
    
    /**
     * Connect to Redis with automatic reconnection
     */
    private function getRedis(): \Redis
    {
        if ($this->redis !== null) {
            try {
                $this->redis->ping();
                return $this->redis;
            } catch (\RedisException $e) {
                $this->logDaemon("Redis connection lost: " . $e->getMessage());
                $this->redis = null;
            }
        }
        
        $retries = 3;
        $delay = 1;
        
        for ($i = 0; $i < $retries; $i++) {
            try {
                $redis = new \Redis();
                $redis->connect($this->redisHost, $this->redisPort, 5.0);
                if ($this->redisDb > 0) {
                    $redis->select($this->redisDb);
                }
                $this->redis = $redis;
                $this->logDaemon("Redis connected");
                return $this->redis;
            } catch (\RedisException $e) {
                $this->logDaemon(sprintf("Redis connect attempt %d failed: %s", $i + 1, $e->getMessage()));
                sleep($delay);
                $delay *= 2;
            }
        }
        
        throw new \RuntimeException("Failed to connect to Redis after $retries attempts");
    }
    
    /**
     * Initialize ZMQ sockets and poll set
     */
    private function initZmq(): void
    {
        $ctx = new \ZMQContext();
        
        // SUB to bitcoind hashblock
        $this->bitcoindSub = new \ZMQSocket($ctx, \ZMQ::SOCKET_SUB);
        $this->bitcoindSub->setSockOpt(\ZMQ::SOCKOPT_SUBSCRIBE, 'hashblock');
        $this->bitcoindSub->setSockOpt(\ZMQ::SOCKOPT_RCVTIMEO, 1000);
        $this->bitcoindSub->connect($this->bitcoindZmqEndpoint);
        $this->logDaemon("ZMQ SUB connected to bitcoind: " . $this->bitcoindZmqEndpoint);
        
        // SUB to SeaTidePool poolstats
        $this->poolstatsSub = new \ZMQSocket($ctx, \ZMQ::SOCKET_SUB);
        $this->poolstatsSub->setSockOpt(\ZMQ::SOCKOPT_SUBSCRIBE, 'poolstats');
        $this->poolstatsSub->setSockOpt(\ZMQ::SOCKOPT_RCVTIMEO, 1000);
        $this->poolstatsSub->connect($this->seatidepoolZmqEndpoint);
        $this->logDaemon("ZMQ SUB connected to SeaTidePool: " . $this->seatidepoolZmqEndpoint);
        
        // Create poll set for multiplexing
        $this->poll = new \ZMQPoll();
        $this->poll->add($this->bitcoindSub, \ZMQ::POLL_IN);
        $this->poll->add($this->poolstatsSub, \ZMQ::POLL_IN);
    }
    
    /**
     * Call bitcoind JSON-RPC
     */
    private function bitcoindRpc(string $method, array $params = []): string|array|null
    {
        $this->incrementMetric('rpc_calls');
        
        $payload = json_encode([
            'jsonrpc' => '1.0',
            'id' => $method,
            'method' => $method,
            'params' => $params,
        ]);
        
        $ch = curl_init($this->bitcoindRpcUrl);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_USERPWD => $this->bitcoindRpcUser . ':' . $this->bitcoindRpcPass,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_CONNECTTIMEOUT => 5,
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($response === false) {
            $this->incrementMetric('rpc_errors');
            $this->logDaemon("RPC error ($method): curl failed — $curlError");
            return null;
        }
        
        if ($httpCode !== 200) {
            $this->incrementMetric('rpc_errors');
            $this->logDaemon("RPC error ($method): HTTP $httpCode");
            return null;
        }
        
        $decoded = json_decode($response, true);
        if (!$decoded || isset($decoded['error']) && $decoded['error'] !== null) {
            $this->incrementMetric('rpc_errors');
            $errMsg = $decoded['error']['message'] ?? 'unknown';
            $this->logDaemon("RPC error ($method): $errMsg");
            return null;
        }
        
        return $decoded['result'] ?? null;
    }
    
    /**
     * Handle bitcoind hashblock notification
     * Polls JSON-RPC for current network stats and writes to Redis
     */
    private function handleHashblock(string $blockHash): void
    {
        $this->incrementMetric('hashblock_received');
        $this->logDaemon(sprintf("hashblock: %s", substr($blockHash, 0, 16) . '...'));
        
        // Get mining info (difficulty, block height, network hashrate)
        $miningInfo = $this->bitcoindRpc('getmininginfo');
        if ($miningInfo === null) {
            return;
        }
        
        // Get network hashrate (nblocks=120 ≈ 20 hours for BTC)
        $nethash = $this->bitcoindRpc('getnetworkhashps', [120]);
        
        $networkData = [
            'height' => $miningInfo['blocks'] ?? 0,
            'difficulty' => $miningInfo['difficulty'] ?? 0,
            'nethash_raw' => $nethash ?? 0,
            'nethash' => $this->formatHashrate($nethash ?? 0),
            'best_block' => $blockHash,
            'chain' => $miningInfo['chain'] ?? 'unknown',
            'updated_at' => date('c'),
        ];
        
        $this->writeRedisHash("chain:{$this->chainId}:network", $networkData);
        
        echo sprintf("[%s] BLOCK height=%d diff=%.2f nethash=%s\n",
            date('H:i:s'),
            $networkData['height'],
            $networkData['difficulty'],
            $networkData['nethash']
        );
    }
    
    /**
     * Handle SeaTidePool poolstats message
     * Writes pool stats directly to Redis
     */
    private function handlePoolstats(string $jsonPayload): void
    {
        $this->incrementMetric('poolstats_received');
        
        $stats = json_decode($jsonPayload, true);
        if (!$stats) {
            $this->logDaemon("Invalid poolstats JSON: " . substr($jsonPayload, 0, 100));
            $this->incrementMetric('errors');
            return;
        }
        
        // Add metadata
        $stats['updated_at'] = date('c');
        
        $this->writeRedisHash("chain:{$this->chainId}:poolstats", $stats);
        
        echo sprintf("[%s] poolstats: workers=%s sps1=%s dsps1=%s\n",
            date('H:i:s'),
            $stats['workers'] ?? '?',
            $stats['SPS1m'] ?? $stats['sps1'] ?? '?',
            $stats['dsps1'] ?? '?'
        );
    }
    
    /**
     * Write an associative array to a Redis hash
     */
    private function writeRedisHash(string $key, array $data): void
    {
        try {
            $redis = $this->getRedis();
            // Flatten nested values to JSON strings for Redis hash storage
            $flat = [];
            foreach ($data as $k => $v) {
                $flat[$k] = is_array($v) ? json_encode($v) : (string)$v;
            }
            $redis->hMSet($key, $flat);
            $this->incrementMetric('redis_writes');
        } catch (\RedisException $e) {
            $this->incrementMetric('redis_errors');
            $this->logDaemon("Redis write error ($key): " . $e->getMessage());
            $this->redis = null;
        }
    }
    
    /**
     * Format hashrate to human-readable string
     */
    private function formatHashrate(float $hashrate): string
    {
        $units = ['H/s', 'KH/s', 'MH/s', 'GH/s', 'TH/s', 'PH/s', 'EH/s', 'ZH/s'];
        $idx = 0;
        while ($hashrate >= 1000 && $idx < count($units) - 1) {
            $hashrate /= 1000;
            $idx++;
        }
        return sprintf('%.2f %s', $hashrate, $units[$idx]);
    }
    
    /**
     * Main run loop
     */
    public function run(): int
    {
        echo APP_NAME . " v" . APP_VERSION . "\n";
        echo str_repeat('=', 40) . "\n";
        echo "Chain: {$this->chainId}\n";
        echo "bitcoind ZMQ: {$this->bitcoindZmqEndpoint}\n";
        echo "bitcoind RPC: {$this->bitcoindRpcUrl}\n";
        echo "SeaTidePool ZMQ: {$this->seatidepoolZmqEndpoint}\n";
        echo "Redis: {$this->redisHost}:{$this->redisPort}/{$this->redisDb}\n";
        echo "PID: " . getmypid() . "\n";
        echo "\n";
        
        // Verify Redis connectivity
        try {
            $this->getRedis();
        } catch (\RuntimeException $e) {
            fprintf(STDERR, "FATAL: %s\n", $e->getMessage());
            return 1;
        }
        
        // Do an initial RPC poll to populate Redis immediately
        $this->logDaemon("Initial network stats poll...");
        $bestHash = $this->bitcoindRpc('getbestblockhash');
        if ($bestHash) {
            $this->handleHashblock($bestHash);
        } else {
            $this->logDaemon("Warning: initial bitcoind RPC failed (will retry on hashblock)");
        }
        
        // Initialize ZMQ sockets
        $this->initZmq();
        echo "Listening for ZMQ messages...\n\n";
        
        $readable = [];
        $writable = [];
        
        while ($this->daemonTick()) {
            try {
                $events = $this->poll->poll($readable, $writable, 1000);
            } catch (\ZMQPollException $e) {
                // EINTR from signal handler — just continue
                if ($e->getCode() === 4) {
                    continue;
                }
                throw $e;
            }
            
            if ($events === 0) {
                continue;
            }
            
            foreach ($readable as $socket) {
                // ZMQ multipart: [envelope, data]
                // bitcoind hashblock: [b"hashblock", raw_block_hash_bytes, sequence]
                // SeaTidePool poolstats: ["poolstats", json_string]
                $envelope = $socket->recv();
                
                if ($socket === $this->bitcoindSub && $envelope === 'hashblock') {
                    $rawHash = $socket->recv();
                    // bitcoind sends raw bytes; convert to hex
                    $blockHash = bin2hex(strrev($rawHash));
                    // Consume sequence number if present
                    if ($socket->getSockOpt(\ZMQ::SOCKOPT_RCVMORE)) {
                        $socket->recv();
                    }
                    $this->handleHashblock($blockHash);
                    
                } elseif ($socket === $this->poolstatsSub && $envelope === 'poolstats') {
                    $payload = $socket->recv();
                    // Consume any remaining frames
                    while ($socket->getSockOpt(\ZMQ::SOCKOPT_RCVMORE)) {
                        $socket->recv();
                    }
                    $this->handlePoolstats($payload);
                }
            }
            
            $this->recordActivity();
        }
        
        $this->cleanupDaemon();
        echo "\nShutdown complete.\n";
        return 0;
    }
}

// --- Main ---

$bridge = new ZmqBridge($settings);
exit($bridge->run());
