#!/usr/bin/env php
<?php
/**
 * TidePoolUI Kafka Share Consumer
 * Reads shares from Kafka topic and stores them in Redis
 * 
 * Uses Enchilada\Daemon\DaemonBehavior trait for:
 * - Graceful shutdown on SIGTERM/SIGINT
 * - Health check file for monitoring
 * - Memory management with periodic GC
 */

declare(ticks=1);

require_once __DIR__ . '/../includes/bootstrap.inc.php';

use Enchilada\Config\IniConfig;

/**
 * Kafka Share Consumer with daemon behavior
 */
class ShareConsumer
{
    use \Enchilada\Daemon\DaemonBehavior;
    
    private string $brokers;
    private string $topic;
    private string $groupId;
    private ?RdKafka\KafkaConsumer $consumer = null;
    private ?ShareStore $store = null;
    private ?IniConfig $settings = null;
    
    public function __construct(?IniConfig $settings = null)
    {
        $this->settings = $settings;
        
        // Settings with env var fallback (env vars take precedence for container/RC script flexibility)
        $this->brokers = getenv('KAFKA_BROKERS') ?: ($settings ? $settings->getString('kafka', 'brokers', 'localhost:9092') : 'localhost:9092');
        $this->topic = getenv('KAFKA_TOPIC') ?: ($settings ? $settings->getString('kafka', 'topic', 'tidepool.dev.shares') : 'tidepool.dev.shares');
        $this->groupId = getenv('KAFKA_GROUP') ?: ($settings ? $settings->getString('kafka', 'group_id', 'tidepoolui-consumer') : 'tidepoolui-consumer');
        
        // Initialize daemon behavior
        $this->initDaemon([
            'health_file' => getenv('HEALTH_FILE') ?: '/tmp/tidepoolui-consumer.health',
            'max_memory_mb' => (int)(getenv('MAX_MEMORY_MB') ?: 128),
        ]);
        
        // Add consumer-specific metrics
        $this->metrics['valid'] = 0;
        $this->metrics['invalid'] = 0;
        $this->metrics['reconnects'] = 0;
        $this->metrics['last_message_at'] = null;
    }
    
    /**
     * Custom health data for this consumer
     */
    protected function getCustomHealthData(): array
    {
        return [
            'kafka' => [
                'brokers' => $this->brokers,
                'topic' => $this->topic,
                'group' => $this->groupId,
            ],
        ];
    }
    
    /**
     * Initialize Kafka consumer
     */
    private function initKafka(): void
    {
        $conf = new RdKafka\Conf();
        $conf->set('group.id', $this->groupId);
        $conf->set('metadata.broker.list', $this->brokers);
        $conf->set('auto.offset.reset', 'earliest');
        $conf->set('enable.auto.commit', 'true');
        $conf->set('session.timeout.ms', '30000');
        $conf->set('heartbeat.interval.ms', '10000');
        
        // Error callback
        $conf->setErrorCb(function ($kafka, $err, $reason) {
            $this->logDaemon(sprintf("Kafka error: %s (reason: %s)", 
                rd_kafka_err2str($err), $reason));
            $this->incrementMetric('errors');
        });
        
        // Rebalance callback
        $conf->setRebalanceCb(function ($kafka, $err, $partitions) {
            switch ($err) {
                case RD_KAFKA_RESP_ERR__ASSIGN_PARTITIONS:
                    $this->logDaemon(sprintf("Partition assignment: %d partitions", count($partitions)));
                    $kafka->assign($partitions);
                    break;
                case RD_KAFKA_RESP_ERR__REVOKE_PARTITIONS:
                    $this->logDaemon("Partition revocation");
                    $kafka->assign(null);
                    break;
                default:
                    $this->logDaemon(sprintf("Rebalance error: %s", rd_kafka_err2str($err)));
            }
        });
        
        $this->consumer = new RdKafka\KafkaConsumer($conf);
        $this->consumer->subscribe([$this->topic]);
    }
    
    /**
     * Get ShareStore with automatic reconnection
     */
    private function getStore(): ShareStore
    {
        if ($this->store !== null) {
            try {
                $this->store->ping();
                return $this->store;
            } catch (Exception $e) {
                $this->logDaemon("Redis connection lost: " . $e->getMessage());
                $this->store = null;
            }
        }
        
        // Reconnect with exponential backoff
        $retries = 3;
        $delay = 1;
        
        for ($i = 0; $i < $retries; $i++) {
            try {
                $redis = tidepoolui_create_redis($this->settings);
                $workerTtl = $this->settings ? $this->settings->getInt('cache', 'worker_ttl', 900) : 900;
                $this->store = new ShareStore($redis, $workerTtl);
                $this->incrementMetric('reconnects');
                $this->logDaemon("Redis reconnected");
                return $this->store;
            } catch (Exception $e) {
                $this->logDaemon(sprintf("Redis reconnect attempt %d failed: %s", $i + 1, $e->getMessage()));
                sleep($delay);
                $delay *= 2;
            }
        }
        
        throw new Exception("Failed to reconnect to Redis after $retries attempts");
    }
    
    /**
     * Process a single share message
     */
    private function processShare(array $share): void
    {
        // Convert createdate to RFC 3339 format
        if (!empty($share['createdate'])) {
            if (strpos($share['createdate'], ',') !== false) {
                list($seconds, $nanos) = explode(',', $share['createdate']);
                $share['createdate'] = date('c', (int)$seconds);
            }
        } else {
            $share['createdate'] = date('c');
        }
        
        $store = $this->getStore();
        $store->addShare($share);
        
        $this->incrementMetric('processed');
        $this->setMetric('last_message_at', date('c'));
        $this->recordActivity();
        
        if (!empty($share['result'])) {
            $this->incrementMetric('valid');
        } else {
            $this->incrementMetric('invalid');
        }
        
        $status = !empty($share['result']) ? 'valid' : 'invalid';
        $worker = $share['workername'] ?? 'unknown';
        echo sprintf("[%s] %s: %s (%s)\n", 
            date('H:i:s'), 
            $worker, 
            $status,
            $share['sdiff'] ?? '?'
        );
    }
    
    /**
     * Main run loop
     */
    public function run(): int
    {
        echo "TidePoolUI Kafka Share Consumer\n";
        echo "================================\n";
        echo "Brokers: {$this->brokers}\n";
        echo "Topic: {$this->topic}\n";
        echo "Group: {$this->groupId}\n";
        echo "PID: " . getmypid() . "\n";
        echo "Health file: {$this->healthFile}\n";
        echo "\n";
        
        $this->initKafka();
        echo "Subscribed to {$this->topic}. Waiting for messages...\n\n";
        
        while ($this->daemonTick()) {
            $message = $this->consumer->consume(1000);
            
            switch ($message->err) {
                case RD_KAFKA_RESP_ERR_NO_ERROR:
                    $share = json_decode($message->payload, true);
                    if ($share) {
                        try {
                            $this->processShare($share);
                        } catch (Exception $e) {
                            $this->logDaemon("Failed to store share: " . $e->getMessage());
                            $this->incrementMetric('errors');
                        }
                    } else {
                        $this->logDaemon("Invalid JSON: " . substr($message->payload, 0, 100));
                        $this->incrementMetric('errors');
                    }
                    break;
                    
                case RD_KAFKA_RESP_ERR__PARTITION_EOF:
                case RD_KAFKA_RESP_ERR__TIMED_OUT:
                    // Normal, continue polling
                    break;
                    
                default:
                    $this->logDaemon("Kafka error: " . $message->errstr());
                    $this->incrementMetric('errors');
                    break;
            }
        }
        
        $this->cleanupDaemon();
        return 0;
    }
}

// Run the consumer
/** @var IniConfig|null $SETTINGS */
global $SETTINGS;
$consumer = new ShareConsumer($SETTINGS);
exit($consumer->run());
