# TidePoolUI MVP Roadmap

This document outlines planned features for the Minimum Viable Product (MVP).

## 1. Block Tracking

**Status:** Planned  
**Kafka Impact:** May require `tidepool.*.blocks` topic subscription

Track solved blocks at pool and worker levels:

- **Pool level:** Total blocks solved, last block height, last block time
- **Worker level:** Blocks solved by each worker, last block info

### Implementation Notes

SeaTidePool already publishes to `tidepool.*.blocks` topic. The consumer needs to:
1. Subscribe to the blocks topic in addition to shares
2. Store block data in Redis (`tidepoolui:blocks`, `tidepoolui:blocks:worker:<name>`)
3. Add API endpoints for block stats
4. Update frontend to display block counts

---

## 2. World Map Visualization

**Status:** Planned  
**Kafka Impact:** None (uses existing GeoIP data)

Live map showing miner locations with minimalist design.

### Design Goals

- Responsive, works on mobile
- Real-time updates (WebSocket or polling)
- Minimalist - don't clutter the dashboard
- Smooth animations for new connections

### Implementation Options

| Library | Size | Notes |
|---------|------|-------|
| **Leaflet** | 40KB | Lightweight, tile-based |
| **MapLibre GL** | 200KB | Vector tiles, smooth |
| **D3.js** | 70KB | Full control, SVG-based |
| **Simple SVG** | <10KB | Custom world SVG, no tiles |

Recommendation: Start with a simple SVG world map for minimalism, upgrade to Leaflet if more features needed.

### Data Flow

```
GeoIP data (already in Redis) → API endpoint → Frontend map component
```

---

## 3. REST API Documentation (OpenAPI)

**Status:** Planned  
**Kafka Impact:** None

Create OpenAPI 3.0 spec and self-hosted documentation viewer.

### Self-Hosted Documentation Tools

| Tool | Language | Notes |
|------|----------|-------|
| **Swagger UI** | JS | Standard, widely used |
| **Redoc** | JS | Clean single-page |
| **RapiDoc** | JS | Modern, customizable |
| **Stoplight Elements** | JS | Beautiful, interactive |

Recommendation: Redoc for clean minimalist look, or Swagger UI for interactivity.

### Implementation

1. Create `openapi.yaml` spec file
2. Add `/api/docs` endpoint serving documentation
3. Include as part of frontend or separate route

---

## 4. Aggregator API Endpoint

**Status:** Planned  
**Kafka Impact:** None

Standard API format for pool aggregator sites (e.g., MiningPoolStats, WhereToMine).

### Common Standards

Most aggregators expect endpoints like:

```
GET /api/stats
{
  "hashrate": 1234567890,
  "workers": 42,
  "fee": 1.0,
  "minpayout": 0.001,
  "lastblock": 123456,
  "coin": "BTC",
  "algorithm": "sha256"
}
```

No formal standard exists, but common fields include:
- `hashrate` (H/s)
- `workers` (count)
- `fee` (percentage)
- `minpayout` (coins)
- `lastblock` (height or timestamp)

### Implementation

Add `/api/v1/aggregator` or `/api/pool` endpoint with standardized response.

---

## 5. Multi-Pool Support

**Status:** Research Required  
**Kafka Impact:** Topic naming already supports multiple pools via prefix

Support multiple pool instances from a single TidePoolUI deployment.

### Architecture Options

#### Option A: URL-Based Config (Enchilada Pattern)

Based on [Enchilada multisite.inc.php](http://git.morante.net/unibia/MultiFaucetRPC):

```
pool1.example.com → config/pool1.example.com/
pool2.example.com → config/pool2.example.com/
```

**Pros:** Simple, proven pattern, low ops burden  
**Cons:** Requires DNS setup per pool, config file management

#### Option B: Path-Based Routing

```
example.com/pool1/ → Pool 1 config
example.com/pool2/ → Pool 2 config
```

**Pros:** Single domain, easier SSL  
**Cons:** Frontend routing complexity

#### Option C: Database-Driven Multi-Tenancy

Store pool configs in database, select by hostname or API key.

**Pros:** Dynamic, no file management  
**Cons:** More complex, requires admin UI

#### Option D: Environment/Container Per Pool

Each pool runs its own TidePoolUI container with different env vars.

**Pros:** Complete isolation, simple code  
**Cons:** Higher resource usage, more containers to manage

### Kafka Considerations

Topic prefix already supports this: `tidepool.pool1.shares`, `tidepool.pool2.shares`

Consumer needs to know which topic(s) to subscribe to per pool.

### Recommendation

Start with **Option A** (Enchilada pattern) for simplicity:
1. Detect hostname
2. Load pool-specific config from `config/<hostname>/`
3. Each config specifies Kafka topic prefix, Redis prefix, etc.

---

## 6. Historical Stats & Grafana

**Status:** Planned (Separate Project?)  
**Kafka Impact:** Consumer for time-series data

Store historical statistics in a time-series database for Grafana dashboards.

### Architecture

```
Kafka → Stats Consumer → TimescaleDB/InfluxDB → Grafana
```

### Time-Series Database Options

| Database | Notes |
|----------|-------|
| **TimescaleDB** | PostgreSQL extension, SQL queries |
| **InfluxDB** | Purpose-built, InfluxQL/Flux |
| **Prometheus** | Pull-based, good for metrics |
| **VictoriaMetrics** | InfluxDB-compatible, efficient |

### Metrics to Track

- Pool hashrate (1m, 5m, 15m, 1h)
- Worker count
- Shares per minute
- Blocks solved
- Per-worker hashrate

### Implementation Options

1. **Separate project:** `TidePoolMetrics` - dedicated consumer writing to TSDB
2. **Grafana plugin:** Direct Kafka → Grafana via plugin
3. **Extend TidePoolUI consumer:** Add TSDB writes alongside Redis

Recommendation: Separate project (`TidePoolMetrics`) for clean separation of concerns.

---

## 7. Real-Time Updates (Scalability)

**Status:** Planned  
**Kafka Impact:** Redis Pub/Sub integration

Current implementation uses HTTP polling (`hx-trigger="every 5s"`). This doesn't scale well:

| Users | Requests/sec | Notes |
|-------|--------------|-------|
| 1,000 | 200 | Manageable |
| 10,000 | 2,000 | High server load |
| 100,000 | 20,000 | Unsustainable |

### Technology Comparison

| Technology | Direction | Browser Support | PHP Support | Best For |
|------------|-----------|-----------------|-------------|----------|
| **HTTP Polling** | Request/Response | 100% | Native | Simple, low traffic |
| **SSE** | Server→Client | 97% | Native | Dashboards, feeds |
| **WebSocket** | Bidirectional | 98% | Ratchet/Swoole | Chat, interactive |
| **WebTransport** | Bidirectional + Datagrams | ~70% | None | Gaming, video, VR |

### WebTransport / HTTP/3 / QUIC Analysis

| Technology | Status | Notes |
|------------|--------|-------|
| **HTTP/3 (QUIC)** | Stable, 95% browser support | Enable at nginx/Caddy level |
| **WebTransport** | Emerging, Chrome/Edge only | No Firefox/Safari, no PHP support |

**WebTransport pros:** Multiplexed streams, unreliable datagrams, lower latency, 0-RTT  
**WebTransport cons:** Incomplete browser support, requires Go/Rust/Node, overkill for stats

### Recommendation: Server-Sent Events (SSE)

For a **read-only dashboard**, SSE is ideal:

```
Current:  Browser ──poll──> PHP ──> Redis ──> Response
          (every 5s, all users, all endpoints)

SSE:      Consumer ──publish──> Redis Pub/Sub
          Browser ──SSE──> PHP ──subscribe──> Redis
          (push only when data changes)
```

**HTMX has native SSE support:**

```html
<div hx-ext="sse" sse-connect="/api/v1/stream" sse-swap="stats">
    <!-- Auto-updates when server pushes "stats" event -->
</div>
```

### Scaling Ladder

| Scale | Solution |
|-------|----------|
| **MVP** | SSE with PHP |
| **10K+ users** | Add Mercure or Centrifugo as broadcast layer |
| **100K+ users** | Dedicated Go/Rust SSE service |
| **Future** | WebTransport when browser support matures |

### Implementation Steps

1. Add Redis Pub/Sub publishing in consumer (on share/block events)
2. Create `/api/v1/stream` SSE endpoint
3. Replace `hx-trigger="every 5s"` with `hx-ext="sse"`
4. Keep REST endpoints for initial page load and non-realtime data

---

## Priority Order

1. **Block Tracking** - Core feature, relatively simple
2. **Aggregator API** - Quick win, enables pool listing
3. **API Documentation** - Developer experience
4. **World Map** - Visual appeal, uses existing data
5. **Real-Time Updates** - Scalability foundation
6. **Multi-Pool Support** - Architecture decision needed
7. **Historical Stats** - Separate project scope

---

## Future Considerations: Message Brokers

Analysis of whether RabbitMQ, MQTT, or ZMQ would benefit the architecture.

### Current Stack

```mermaid
graph LR
    A[SeaTidePool] -->|shares| B[Kafka]
    B --> C[Consumer]
    C --> D[Redis]
    D -->|Pub/Sub| E[SSE Workers]
    D -->|Cache| F[REST API]
    E --> G[Browsers]
    F --> G
```

**Redis serves dual purposes:**
- **Pub/Sub** - Real-time streaming to SSE workers
- **Cache/Store** - Pool stats, worker data, GeoIP (non-streamed data)

### Broker Comparison

| Broker | Strength | In Ecosystem? | TidePoolUI Fit |
|--------|----------|---------------|----------------|
| **Kafka** | Event streaming, durability | ✅ SeaTidePool | Core backbone |
| **ZMQ** | Low-latency IPC, brokerless | ✅ Bitcoin→SeaTidePool | Alternative to Redis Pub/Sub |
| **RabbitMQ** | Task queues, routing | ❌ | Redundant with Kafka |
| **MQTT** | IoT, lightweight | ❌ | Miners use Stratum |

### ZMQ Consideration

SeaTidePool already uses ZMQ for Bitcoin block notifications (`zmqblock`). ZMQ could replace Redis Pub/Sub for SSE fan-out:

```
Consumer ──ZMQ PUB──> SSE Workers (no broker, direct IPC)
```

**Pros:** Lower latency, no central broker, already a dependency  
**Cons:** PHP ZMQ extension less maintained, adds complexity

### Recommendation

**Keep current stack:**
- **Kafka** - Share/block event ingestion
- **Redis Pub/Sub** - SSE broadcast layer
- **Redis Cache** - Stats, worker data, GeoIP (non-streamed)

**Consider ZMQ** if Redis Pub/Sub becomes a bottleneck at 100K+ concurrent SSE connections.

---

## Related Links

- [SeaTidePool Kafka Topics](../doc/KAFKA_CONSUMER.md)
- [Enchilada Multisite Module](http://git.morante.net/unibia/MultiFaucetRPC/raw/branch/master/system/multisite.inc.php)
- [OpenAPI Specification](https://swagger.io/specification/)
- [ZeroMQ Guide](https://zguide.zeromq.org/)
