# TidePoolUI FreeBSD Installation Runbook

Complete guide for deploying TidePoolUI on a fresh FreeBSD system with Apache web server.

## Prerequisites

- FreeBSD 14.x or later
- Root access
- Network connectivity to:
  - Your Kafka cluster
  - Redis server (local or remote)
  - SeaTidePool mining pool server

## Architecture Overview

```
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│   SeaTidePool   │────▶│  Kafka Cluster  │────▶│  TidePoolUI     │
│  (Mining Pool)  │     │                 │     │  Consumer       │
└─────────────────┘     └─────────────────┘     └────────┬────────┘
                                                         │
                                                         ▼
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│    Browser      │────▶│  Apache + PHP   │────▶│     Redis       │
│                 │     │  (Frontend+API) │     │                 │
└─────────────────┘     └─────────────────┘     └─────────────────┘
```

---

## Part 1: FreeBSD System Setup

### 1.1 Update System

```sh
freebsd-update fetch install
pkg update && pkg upgrade -y
```

### 1.2 Install Required Packages

```sh
pkg install -y \
    apache24 \
    php83 \
    php83-extensions \
    php83-filter \
    php83-json \
    php83-pcntl \
    php83-posix \
    php83-redis \
    mod_php83 \
    redis \
    pecl-rdkafka
```

### 1.3 Enable Services

```sh
sysrc apache24_enable="YES"
sysrc redis_enable="YES"
sysrc php_fpm_enable="YES"
```

---

## Part 2: Redis Configuration

### 2.1 Configure Redis

Edit `/usr/local/etc/redis.conf`:

```sh
# Bind to localhost only (or your internal network)
bind 127.0.0.1

# Set memory limit
maxmemory 256mb
maxmemory-policy allkeys-lru

# Persistence (optional, data is ephemeral for TidePoolUI)
save ""
```

### 2.2 Start Redis

```sh
service redis start
redis-cli ping  # Should return PONG
```

---

## Part 3: TidePoolUI Installation

### 3.1 Add Morante Ports Tree (if using ports)

```sh
# Clone the Morante ports overlay
mkdir -p /usr/local/poudriere/ports
git clone https://git.morante.com/FreeBSD/ports.git /usr/local/poudriere/ports/morante
```

Or install directly from packages (when available):

```sh
pkg install tidepoolui-frontend tidepoolui-api tidepoolui-consumer
```

### 3.2 Manual Installation (Alternative)

If ports are not yet in your tree:

```sh
# Download release
fetch https://git.morante.com/TidePool/TidePoolUI/-/archive/v0.1.0/TidePoolUI-v0.1.0.tar.gz
tar -xzf TidePoolUI-v0.1.0.tar.gz
cd TidePoolUI-v0.1.0

# Install frontend
mkdir -p /usr/local/www/tidepoolui/frontend
cp -r frontend/* /usr/local/www/tidepoolui/frontend/
cp frontend/.htaccess /usr/local/www/tidepoolui/frontend/

# Install API
mkdir -p /usr/local/www/tidepoolui/backend
cp -r backend/api backend/classes backend/config backend/includes \
      backend/libraries backend/system /usr/local/www/tidepoolui/backend/
cp backend/.htaccess /usr/local/www/tidepoolui/backend/

# Install consumer
mkdir -p /usr/local/libexec/tidepoolui
cp backend/bin/share-consumer.php /usr/local/libexec/tidepoolui/
cp -r backend/classes backend/config backend/includes \
      backend/libraries backend/system /usr/local/libexec/tidepoolui/

# Set ownership
chown -R www:www /usr/local/www/tidepoolui
chown -R www:www /usr/local/libexec/tidepoolui
```

---

## Part 4: Apache Configuration

### 4.1 Enable PHP Module

Edit `/usr/local/etc/apache24/httpd.conf`:

```apache
# Uncomment or add:
LoadModule php_module libexec/apache24/libphp.so

# Add PHP handler
<FilesMatch "\.php$">
    SetHandler application/x-httpd-php
</FilesMatch>

# Add index.php to DirectoryIndex
<IfModule dir_module>
    DirectoryIndex index.php index.html
</IfModule>
```

### 4.2 Create VirtualHost

Create `/usr/local/etc/apache24/Includes/tidepoolui.conf`:

```apache
<VirtualHost *:80>
    ServerName tidepoolui.example.com
    DocumentRoot "/usr/local/www/tidepoolui/frontend"

    # Frontend
    <Directory "/usr/local/www/tidepoolui/frontend">
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>

    # API proxy to backend
    Alias /api "/usr/local/www/tidepoolui/backend/api"
    <Directory "/usr/local/www/tidepoolui/backend/api">
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted

        # PHP settings for API
        <FilesMatch "\.php$">
            SetHandler application/x-httpd-php
        </FilesMatch>
    </Directory>

    # Backend classes/config (deny direct access)
    <Directory "/usr/local/www/tidepoolui/backend/classes">
        Require all denied
    </Directory>
    <Directory "/usr/local/www/tidepoolui/backend/config">
        Require all denied
    </Directory>

    # Logging
    ErrorLog "/var/log/tidepoolui-error.log"
    CustomLog "/var/log/tidepoolui-access.log" combined
</VirtualHost>
```

### 4.3 Enable mod_rewrite

```sh
# Edit httpd.conf and uncomment:
LoadModule rewrite_module libexec/apache24/mod_rewrite.so
```

### 4.4 Test and Start Apache

```sh
apachectl configtest
service apache24 start
```

---

## Part 5: TidePoolUI Configuration

### 5.1 Configure API

```sh
cd /usr/local/www/tidepoolui/backend/config
cp settings.ini.sample settings.ini
```

Edit `settings.ini`:

```ini
[redis]
host = 127.0.0.1
port = 6379
db = 0

[cache]
worker_ttl = 900   ; 15 minutes - workers disappear after no shares

[api]
cors_origins = *

[app]
demo_enabled = false
```

### 5.2 Configure Consumer

Copy the sample environment file and edit it:

```sh
cp /usr/local/etc/tidepoolui/consumer.env.sample /usr/local/etc/tidepoolui/consumer.env
```

Edit `/usr/local/etc/tidepoolui/consumer.env`:

```sh
# Kafka
KAFKA_BROKERS="kafka1.example.com:9092,kafka2.example.com:9092"
KAFKA_TOPIC="tidepool.prod.shares"
KAFKA_GROUP="tidepoolui-consumer"

# Redis
REDIS_HOST="127.0.0.1"
REDIS_PORT="6379"
REDIS_DB="0"

# Daemon
HEALTH_FILE="/var/run/tidepoolui-consumer.health"
MAX_MEMORY_MB="128"
```

Enable the service in `/etc/rc.conf`:

```sh
tidepoolui_consumer_enable="YES"
```

**Multi-chain setup:** Each chain needs its own consumer instance with a separate env file and Redis database. For example, Bitcoin on `REDIS_DB="0"` and Litecoin on `REDIS_DB="1"`:

```sh
# /usr/local/etc/tidepoolui/consumer-bitcoin.env
KAFKA_TOPIC="bitcoin.prod.shares"
KAFKA_GROUP="bitcoin-ui-consumer"
REDIS_DB="0"

# /usr/local/etc/tidepoolui/consumer-litecoin.env
KAFKA_TOPIC="litecoin.prod.shares"
KAFKA_GROUP="litecoin-ui-consumer"
REDIS_DB="1"
```

Each chain's API backend Apache vhost should also set `SetEnv REDIS_DB "N"` to match.

### 5.3 Install Consumer RC Script

If not installed via ports:

```sh
cp /path/to/TidePoolUI/etc/rc.d/tidepoolui_consumer /usr/local/etc/rc.d/
chmod +x /usr/local/etc/rc.d/tidepoolui_consumer
cp /path/to/TidePoolUI/etc/consumer.env.sample /usr/local/etc/tidepoolui/consumer.env
```

### 5.4 Start Consumer

```sh
service tidepoolui_consumer start
service tidepoolui_consumer status
```

---

## Part 6: Kafka Cluster Configuration

These steps are performed on your **existing Kafka cluster**.

### 6.1 Create Topic for Shares

```sh
# On your Kafka broker
kafka-topics.sh --create \
    --bootstrap-server localhost:9092 \
    --topic tidepool.prod.shares \
    --partitions 3 \
    --replication-factor 2 \
    --config retention.ms=86400000  # 24 hours
```

### 6.2 Verify Topic

```sh
kafka-topics.sh --describe \
    --bootstrap-server localhost:9092 \
    --topic tidepool.prod.shares
```

### 6.3 Configure ACLs (if using authentication)

```sh
# Allow SeaTidePool to produce
kafka-acls.sh --bootstrap-server localhost:9092 \
    --add --allow-principal User:seatidepool \
    --producer --topic tidepool.prod.shares

# Allow TidePoolUI consumer to consume
kafka-acls.sh --bootstrap-server localhost:9092 \
    --add --allow-principal User:tidepoolui \
    --consumer --topic tidepool.prod.shares \
    --group tidepoolui-consumer
```

---

## Part 7: SeaTidePool Configuration

These steps are performed on your **SeaTidePool mining pool server**.

### 7.1 Enable Kafka in SeaTidePool

Edit your SeaTidePool configuration (`tidepool.conf`):

```ini
[kafka]
enabled = true
brokers = kafka1.example.com:9092,kafka2.example.com:9092
topic = tidepool.prod.shares

# Optional: authentication
#sasl_mechanism = PLAIN
#sasl_username = seatidepool
#sasl_password = your-password
```

### 7.2 Restart SeaTidePool

```sh
service seatidepool restart
```

### 7.3 Verify Kafka Publishing

Check SeaTidePool logs for Kafka connection:

```sh
tail -f /var/log/seatidepool/pool.log | grep -i kafka
```

You should see messages like:
```
Kafka producer connected to kafka1.example.com:9092
Publishing share to topic tidepool.prod.shares
```

---

## Part 8: Verification

### 8.1 Check All Services

```sh
# Apache
service apache24 status
curl -I http://localhost/

# Redis
redis-cli ping

# Consumer
service tidepoolui_consumer status
service tidepoolui_consumer health
```

### 8.2 Test API Endpoints

```sh
# Health check
curl http://localhost/api/v1/health

# Stats (will be empty initially)
curl http://localhost/api/v1/stats

# Workers
curl http://localhost/api/v1/workers
```

### 8.3 Monitor Kafka Consumer

```sh
# Watch consumer lag
kafka-consumer-groups.sh --bootstrap-server kafka1.example.com:9092 \
    --describe --group tidepoolui-consumer

# Check consumer health file
cat /var/run/tidepoolui-consumer.health
```

### 8.4 Test End-to-End

1. Submit a share to SeaTidePool (mine with a worker)
2. Check Kafka topic has the message:
   ```sh
   kafka-console-consumer.sh --bootstrap-server kafka1.example.com:9092 \
       --topic tidepool.prod.shares --from-beginning --max-messages 1
   ```
3. Check Redis has the data:
   ```sh
   redis-cli keys "tidepoolui:*"
   redis-cli hgetall "tidepoolui:workers:yourworker"
   ```
4. Check the web UI shows the worker

---

## Part 9: Troubleshooting

### Consumer Not Starting

```sh
# Check logs
tail -f /var/log/messages | grep tidepoolui

# Run manually for debugging
/usr/local/bin/php /usr/local/libexec/tidepoolui/share-consumer.php
```

### No Data in Redis

1. Verify Kafka connection:
   ```sh
   # Test connectivity
   nc -zv kafka1.example.com 9092
   ```

2. Check consumer is receiving messages:
   ```sh
   service tidepoolui_consumer health
   # Look at "processed" counter
   ```

3. Verify SeaTidePool is publishing:
   ```sh
   kafka-console-consumer.sh --bootstrap-server kafka1.example.com:9092 \
       --topic tidepool.prod.shares --timeout-ms 5000
   ```

### API Returns Errors

```sh
# Check PHP errors
tail -f /var/log/tidepoolui-error.log

# Test Redis connectivity from PHP
php -r "
\$r = new Redis();
\$r->connect('127.0.0.1', 6379);
var_dump(\$r->ping());
"
```

### Workers Disappearing Too Fast

Increase `worker_ttl` in `/usr/local/www/tidepoolui/backend/config/settings.ini`:

```ini
[cache]
worker_ttl = 1800   ; 30 minutes
```

---

## Part 10: Security Hardening

### 10.1 Firewall Rules

```sh
# Allow HTTP/HTTPS
ipfw add allow tcp from any to me 80
ipfw add allow tcp from any to me 443

# Block Redis from external
ipfw add deny tcp from any to me 6379

# Allow Kafka from internal network only
ipfw add allow tcp from 10.0.0.0/8 to me 9092
```

### 10.2 Enable HTTPS

```sh
pkg install py311-certbot py311-certbot-apache

certbot --apache -d tidepoolui.example.com
```

### 10.3 Redis Authentication

Edit `/usr/local/etc/redis.conf`:

```
requirepass your-strong-password
```

Update TidePoolUI config to use password.

---

## Part 11: Production API Gateway (HAProxy + SSI)

In production, TidePoolUI uses an HAProxy API gateway at `api.securepayment.cc` to route requests to per-chain backends. The frontend at `pool.securepayment.cc` uses Apache SSI to inject the gateway URL at serve-time.

### 11.1 Apache Frontend VHost (SSI)

Update the frontend vhost to enable SSI and inject the gateway base URL:

```apache
<VirtualHost *:8001>
    ServerName pool.securepayment.cc
    DocumentRoot "/usr/local/www/tidepoolui/frontend"

    # Enable SSI on .html files
    AddOutputFilter INCLUDES .html

    <Directory "/usr/local/www/tidepoolui/frontend">
        Options -Indexes +FollowSymLinks +Includes
        AllowOverride All
        Require all granted
    </Directory>

    # API gateway base URL injected via SSI
    SetEnv API_GATEWAY "https://api.securepayment.cc/mining"
</VirtualHost>
```

**Note**: If `SetEnv API_GATEWAY` is omitted or empty, the frontend falls back to same-origin `/api/v1` (dev mode behavior).

### 11.2 HAProxy Chain Discovery

HAProxy serves a static JSON file for chain discovery. Create `/usr/local/etc/haproxy.d/chains.json`:

```json
{
    "chains": [
        {"id": "bitcoin", "name": "Bitcoin"},
        {"id": "litecoin", "name": "Litecoin"}
    ],
    "default": "bitcoin"
}
```

The frontend fetches this on load to populate the chain selector dropdown.

### 11.3 HAProxy Configuration Summary

The full HAProxy config is in the architecture plan (`configurable-api-base-url-4448fb.md`). Key sections:

- **DNS resolver** (`resolvers dns`) — for SRV record discovery
- **CORS preflight** — returns 204 for `OPTIONS` on `api.securepayment.cc`
- **Chain discovery** — serves `chains.json` at `GET /mining/`
- **Per-chain backends** (`bk_mining_bitcoin`, `bk_mining_litecoin`) — path rewrite `/mining/{chain}/v1/...` → `/api/v1/...`, CORS headers, SRV-based server discovery
- **Direct debug backends** (`api_bitcoin`, `api_litecoin`) — no path rewrite, accessed via `bitcoin.api.securepayment.cc`

### 11.4 Domain-to-Backend Map

Add entries to `/usr/local/etc/haproxy.d/domain2backend.db`:

```
bitcoin.api.securepayment.cc   api_bitcoin
litecoin.api.securepayment.cc  api_litecoin
```

### 11.5 DNS SRV Records

Each backend requires an SRV record for HAProxy's `server-template` discovery:

| Record | Example Resolution |
|---|---|
| `_http._tcp.app.securepayment.cc` | `10 5 8001 app01.securepayment.cc.` |
| `_http._tcp.bitcoin.api.securepayment.cc` | `10 5 8001 api01.securepayment.cc.` |
| `_http._tcp.litecoin.api.securepayment.cc` | `10 5 8001 api02.securepayment.cc.` |

### 11.6 Adding a New Chain

1. Edit `/usr/local/etc/haproxy.d/chains.json` — add chain entry
2. Add HAProxy ACL + `use_backend` line + new backend block → `service haproxy reload`
3. Add DNS SRV record `_http._tcp.{chain}.api.securepayment.cc`
4. No Apache, frontend, or backend code changes required

---

## Quick Reference

| Component | Config File | Service |
|-----------|-------------|---------|
| Apache | `/usr/local/etc/apache24/` | `apache24` |
| Redis | `/usr/local/etc/redis.conf` | `redis` |
| API | `/usr/local/www/tidepoolui/backend/config/settings.ini` | - |
| Consumer | `/etc/rc.conf` | `tidepoolui_consumer` |
| SeaTidePool | `/path/to/tidepool.conf` | `seatidepool` |

| Log File | Purpose |
|----------|---------|
| `/var/log/tidepoolui-error.log` | Apache/PHP errors |
| `/var/log/tidepoolui-access.log` | HTTP access log |
| `/var/log/messages` | Consumer daemon logs |
| `/var/run/tidepoolui-consumer.health` | Consumer health JSON |
