# TidePoolUI API Reference

This document describes the REST API endpoints provided by TidePoolUI.

## Base URL

- Development: `http://localhost:8080/api/v1`
- Production: `https://your-domain/api/v1`

## Authentication

Currently, the API does not require authentication. It is intended for internal network use.

## Endpoints

### GET /health

Health check endpoint.

**Response:**
```json
{
    "status": "ok",
    "timestamp": "2026-01-28T05:15:00Z"
}
```

---

### GET /stats

Returns aggregate pool statistics.

**Response:**
```json
{
    "success": true,
    "data": {
        "total_shares": 1234,
        "valid_shares": 1200,
        "invalid_shares": 34,
        "workers_seen": 5,
        "acceptance_rate": 97.24,
        "shares_per_minute": 12.5
    }
}
```

**Fields:**
| Field | Type | Description |
|-------|------|-------------|
| `total_shares` | integer | Total shares received |
| `valid_shares` | integer | Shares accepted by pool |
| `invalid_shares` | integer | Shares rejected |
| `workers_seen` | integer | Unique worker count |
| `acceptance_rate` | float | Percentage of valid shares |
| `shares_per_minute` | float | Recent share rate |

---

### GET /workers

Returns statistics for all active workers.

**Response:**
```json
{
    "workers": [
        {
            "workername": "miner1.rig1",
            "username": "miner1",
            "shares": "150",
            "valid": "148",
            "invalid": "2",
            "last_diff": "2.5",
            "total_diff": "375.5",
            "address": "192.168.1.100",
            "agent": "bfgminer/5.5.0",
            "last_seen": "2026-01-28 05:15:00"
        }
    ]
}
```

**Worker Fields:**
| Field | Type | Description |
|-------|------|-------------|
| `workername` | string | Full worker identifier |
| `username` | string | Pool username |
| `shares` | string | Total shares submitted |
| `valid` | string | Valid shares count |
| `invalid` | string | Invalid shares count |
| `last_diff` | string | Difficulty of last share |
| `total_diff` | string | Cumulative difficulty |
| `address` | string | Client IP address |
| `agent` | string | Mining software identifier |
| `last_seen` | string | Timestamp of last share |

---

### GET /shares

Returns recent shares.

**Query Parameters:**
| Parameter | Type | Default | Max | Description |
|-----------|------|---------|-----|-------------|
| `limit` | integer | 50 | 100 | Number of shares to return |

**Example:**
```sh
curl "http://localhost:8080/api/v1/shares?limit=10"
```

**Response:**
```json
{
    "shares": [
        {
            "workinfoid": "12345",
            "clientid": "1",
            "diff": "1.0",
            "sdiff": "2.5",
            "result": true,
            "username": "miner1",
            "workername": "miner1.rig1",
            "address": "192.168.1.100",
            "agent": "bfgminer/5.5.0",
            "createdate": "2026-01-28 05:15:00"
        }
    ]
}
```

**Share Fields:**
| Field | Type | Description |
|-------|------|-------------|
| `workinfoid` | string | Work unit identifier |
| `clientid` | string | Client connection ID |
| `diff` | string | Target difficulty |
| `sdiff` | string | Share difficulty |
| `result` | boolean | true = valid, false = invalid |
| `errn` | integer | Error number (if invalid) |
| `username` | string | Pool username |
| `workername` | string | Full worker identifier |
| `address` | string | Client IP address |
| `agent` | string | Mining software |
| `createdate` | string | Timestamp |

---

### POST /demo

Adds demo/test data for UI development.

**Request:**
No body required.

**Response:**
```json
{
    "success": true,
    "message": "Demo data added",
    "data": {
        "shares_added": 10,
        "workers_added": 3
    }
}
```

---

### DELETE /data

Clears all stored data (development only).

**Response:**
```json
{
    "success": true,
    "message": "All data cleared"
}
```

## Error Responses

All errors follow this format:

```json
{
    "success": false,
    "error": "Error message here",
    "code": 400
}
```

**Common Error Codes:**
| Code | Description |
|------|-------------|
| 400 | Bad Request - Invalid parameters |
| 404 | Not Found - Endpoint doesn't exist |
| 500 | Internal Server Error |

## Rate Limiting

No rate limiting is currently implemented. For production use, consider adding rate limits via a reverse proxy.

## CORS

CORS headers are set to allow requests from any origin during development. Restrict this in production.

## Examples

### Shell (curl)

```sh
# Get pool stats
curl -s http://localhost:8080/api/v1/stats | jq

# Get workers
curl -s http://localhost:8080/api/v1/workers | jq '.workers[]'

# Get last 5 shares
curl -s "http://localhost:8080/api/v1/shares?limit=5" | jq '.shares'

# Add demo data
curl -X POST http://localhost:8080/api/v1/demo
```

### JavaScript (fetch)

```javascript
// Get workers
const response = await fetch('/api/v1/workers');
const data = await response.json();
data.workers.forEach(worker => {
    console.log(`${worker.workername}: ${worker.shares} shares`);
});
```

### PHP

```php
$stats = json_decode(file_get_contents('http://localhost:8080/api/v1/stats'), true);
echo "Total shares: " . $stats['data']['total_shares'];
```
