# HAProxy API Gateway

Production deployment guide for the TidePoolUI HAProxy API gateway at `api.securepayment.cc`. The gateway provides path-based routing to per-chain API backends, CORS handling, chain discovery, and SRV-based server discovery.

## Architecture

```
Browser → pools.securepayment.cc (Apache + SSI)
       → Chain selector: [Bitcoin ▾]
       → api.securepayment.cc/mining/bitcoin/v1/stats
            → HAProxy (CORS + path rewrite + SRV discovery)
            → /api/v1/stats → api01.securepayment.cc:8001

Direct debug (bypass gateway):
  curl → bitcoin.api.securepayment.cc/api/v1/stats → backend directly
```

**Key behaviors:**

- **Path rewrite**: `/mining/bitcoin/v1/stats` → `/api/v1/stats` (backend sees standard paths)
- **CORS**: HAProxy handles preflight and response headers (frontend is cross-origin)
- **Chain discovery**: `GET /mining/` returns available chains from a static JSON file
- **SRV discovery**: All backends resolve host+port from DNS SRV records automatically

---

## Prerequisites

### DNS SRV Records

Each backend requires an SRV record for HAProxy's `server-template` discovery:

| Record | Example Resolution |
|---|---|
| `_http._tcp.app.securepayment.cc` | `10 5 8001 app01.securepayment.cc.` |
| `_http._tcp.bitcoin.api.securepayment.cc` | `10 5 8001 api01.securepayment.cc.` |
| `_http._tcp.litecoin.api.securepayment.cc` | `10 5 8001 api02.securepayment.cc.` |

Verify with:

```sh
drill SRV _http._tcp.bitcoin.api.securepayment.cc
```

### Domains

| Domain | Purpose |
|--------|---------|
| `pools.securepayment.cc` | Frontend (Apache + SSI) |
| `api.securepayment.cc` | API gateway (HAProxy, path-based routing) |
| `bitcoin.api.securepayment.cc` | Direct API debug access (bypasses gateway) |
| `litecoin.api.securepayment.cc` | Direct API debug access (bypasses gateway) |

---

## HAProxy Configuration

### Complete `haproxy.conf`

```haproxy
global
    daemon
    log /dev/log local2 info

resolvers dns
    nameserver ns1 127.0.0.1:53
    resolve_retries 3
    timeout resolve 1s
    timeout retry 1s
    hold valid 30s
    accepted_payload_size 8192

defaults
    log global
    option forwardfor
    mode http
    timeout connect 5000ms
    timeout client 50000ms
    timeout server 50000ms

frontend stats
    bind :8181
    stats uri /
    stats show-modules

frontend CacheFrontend
    bind    *:8080 accept-proxy
    http-response add-header X-Realm-Cache "Naples, FL"

    # API gateway ACLs
    acl is_api_host      req.hdr(host) -i api.securepayment.cc
    acl path_api_root    path /
    acl path_mining_root path /mining /mining/
    acl path_mining_btc  path_beg /mining/bitcoin/
    acl path_mining_ltc  path_beg /mining/litecoin/

    # CORS preflight
    http-request return status 204 hdr Access-Control-Allow-Origin "https://pools.securepayment.cc" hdr Access-Control-Allow-Methods "GET, POST, OPTIONS" hdr Access-Control-Allow-Headers "Content-Type, X-Admin-Secret, HX-Request, HX-Current-URL, HX-Target, HX-Trigger" hdr Access-Control-Max-Age "86400" if is_api_host METH_OPTIONS

    # API root — nothing to see here
    http-request return status 404 content-type "text/html; charset=utf-8" file /usr/local/etc/haproxy.d/api-404.html if is_api_host path_api_root

    # Chain discovery — served from static file, no backend needed
    http-request return status 200 content-type "application/json" hdr Access-Control-Allow-Origin "https://pools.securepayment.cc" file /usr/local/etc/haproxy.d/chains.json if is_api_host path_mining_root

    # Per-chain API routing
    use_backend bk_mining_bitcoin  if is_api_host path_mining_btc
    use_backend bk_mining_litecoin if is_api_host path_mining_ltc

    # Domain map fallback (existing behavior)
    use_backend %[req.hdr(host),lower,map_dom(/usr/local/etc/haproxy.d/domain2backend.db,default)]

backend default
    server stats localhost:8181

# Frontend — pools.securepayment.cc (SRV: _http._tcp.app.securepayment.cc)
backend app
    server-template srv 3 _http._tcp.app.securepayment.cc resolvers dns check send-proxy-v2

# Direct API access — bitcoin.api.securepayment.cc (domain map → api_bitcoin)
backend api_bitcoin
    server-template srv 3 _http._tcp.bitcoin.api.securepayment.cc resolvers dns check send-proxy-v2

# Direct API access — litecoin.api.securepayment.cc (domain map → api_litecoin)
backend api_litecoin
    server-template srv 3 _http._tcp.litecoin.api.securepayment.cc resolvers dns check send-proxy-v2

# API gateway — api.securepayment.cc/mining/bitcoin/ → rewrites to /api/v1/...
backend bk_mining_bitcoin
    http-request set-header Host bitcoin.api.securepayment.cc
    http-request set-path /api%[path,regsub(^/mining/bitcoin,,)]
    http-response del-header Access-Control-Allow-Origin
    http-response set-header Access-Control-Allow-Origin "https://pools.securepayment.cc"
    http-response set-header Access-Control-Allow-Methods "GET, POST, OPTIONS"
    http-response set-header Access-Control-Allow-Headers "Content-Type, X-Admin-Secret, HX-Request, HX-Current-URL, HX-Target, HX-Trigger"
    http-response set-header Access-Control-Max-Age "86400"
    server-template srv 3 _http._tcp.bitcoin.api.securepayment.cc resolvers dns check send-proxy-v2

# API gateway — api.securepayment.cc/mining/litecoin/ → rewrites to /api/v1/...
backend bk_mining_litecoin
    http-request set-header Host litecoin.api.securepayment.cc
    http-request set-path /api%[path,regsub(^/mining/litecoin,,)]
    http-response del-header Access-Control-Allow-Origin
    http-response set-header Access-Control-Allow-Origin "https://pools.securepayment.cc"
    http-response set-header Access-Control-Allow-Methods "GET, POST, OPTIONS"
    http-response set-header Access-Control-Allow-Headers "Content-Type, X-Admin-Secret, HX-Request, HX-Current-URL, HX-Target, HX-Trigger"
    http-response set-header Access-Control-Max-Age "86400"
    server-template srv 3 _http._tcp.litecoin.api.securepayment.cc resolvers dns check send-proxy-v2
```

### Key Sections Explained

**`resolvers dns`** — Enables DNS-based service discovery. HAProxy re-queries SRV records every 30 seconds (`hold valid 30s`), automatically picking up new backends.

**`server-template srv 3`** — Creates up to 3 server slots per backend. HAProxy fills them from SRV records. When a new jail/container is added, just add an SRV record — HAProxy discovers it within 30 seconds.

**`http-request set-header Host`** — Rewrites the `Host` header to the chain's direct-access domain before forwarding. This means the backend Apache vhost only needs its own `ServerName` — no `ServerAlias` required for gateway domains.

**`http-request set-path`** — Path rewrite strips the chain prefix and prepends `/api`:
- `/mining/bitcoin/v1/stats` → `/api/v1/stats`
- `/mining/litecoin/v1/workers` → `/api/v1/workers`

The backend receives standard `/api/v1/...` paths and requires zero code changes.

**Direct debug backends** (`api_bitcoin`, `api_litecoin`) — No path rewrite or CORS. Access directly at `bitcoin.api.securepayment.cc/api/v1/stats` for debugging.

---

## Chain Discovery Endpoint

HAProxy serves a static JSON file at `GET https://api.securepayment.cc/mining/`. The frontend fetches this on load to populate the chain selector dropdown.

### `/usr/local/etc/haproxy.d/chains.json`

```json
{
    "chains": [
        {"id": "bitcoin", "name": "Bitcoin"},
        {"id": "litecoin", "name": "Litecoin"}
    ],
    "default": "bitcoin"
}
```

The `id` field maps to the path segment (`/mining/{id}/`). The `default` field determines which chain is selected on first visit.

### `/usr/local/etc/haproxy.d/api-404.html`

```html
<!DOCTYPE html>
<html>
<head><meta charset="utf-8"></head>
<body style="font-family:monospace;text-align:center;padding:4em;color:#ccc;background:#1a1a2e">
<p>Waves carry the hash,<br>
blocks rise from the ocean floor—<br>
the tide pool confirms.</p>
<p style="color:#555">🌊 404</p>
</body>
</html>
```

Served by HAProxy at the API root (`GET https://api.securepayment.cc/`). No backend is involved.

---

## Domain-to-Backend Map

### `/usr/local/etc/haproxy.d/domain2backend.db`

```
pools.securepayment.cc         app
bitcoin.api.securepayment.cc  api_bitcoin
litecoin.api.securepayment.cc api_litecoin
```

This file maps hostnames to HAProxy backends for the `map_dom()` fallback in the frontend section.

---

## Apache Frontend VHost (SSI)

The frontend Apache vhost enables SSI to inject the gateway base URL at serve-time:

```apache
<VirtualHost *:8001>
    ServerName pools.securepayment.cc
    DocumentRoot "/usr/local/www/tidepoolui/frontend"

    # Enable SSI on .html files
    AddOutputFilter INCLUDES .html

    <Directory "/usr/local/www/tidepoolui/frontend">
        Options -Indexes +FollowSymLinks +Includes
        AllowOverride All
        Require all granted
    </Directory>

    # API gateway base URL injected via SSI
    SetEnv API_GATEWAY "https://api.securepayment.cc/mining"
</VirtualHost>
```

If `SetEnv API_GATEWAY` is omitted or empty, the frontend falls back to same-origin `/api/v1` (dev mode behavior).

---

## Adding a New Chain

1. **`chains.json`** — Add entry to `/usr/local/etc/haproxy.d/chains.json`
2. **HAProxy ACL** — Add `acl path_mining_xxx path_beg /mining/{chain}/` to `CacheFrontend`
3. **HAProxy backend** — Add `bk_mining_{chain}` block (copy existing, change chain name in `set-path` and SRV record)
4. **Direct debug backend** — Add `api_{chain}` block with SRV record
5. **Domain map** — Add `{chain}.api.securepayment.cc api_{chain}` to `domain2backend.db`
6. **DNS** — Add SRV record `_http._tcp.{chain}.api.securepayment.cc`
7. **Reload** — `service haproxy reload`

No Apache, frontend, or backend code changes required.

---

## Varnish Configuration

Varnish sits in front of HAProxy and terminates TLS. The default configuration works, but the `X-Admin-Secret` header requires special handling to prevent admin responses from being cached and served to non-admin users.

### Recommended VCL

```vcl
vcl 4.0;

backend default {
    .host = "127.0.0.1";
    .port = "8080";
    .proxy_header = 2;
}

sub vcl_recv {
    # Never cache admin requests — prevent admin-only data leaking to other users
    if (req.http.X-Admin-Secret) {
        return (pass);
    }
}
```

### Caching Considerations

| Request Type | Cached? | Notes |
|---|---|---|
| Normal API requests (`/api/v1/stats`) | Yes | Safe — same data for all users |
| Admin requests (`X-Admin-Secret` header) | **No** | `pass` rule bypasses cache |
| CORS preflight (`OPTIONS`) | Yes | Safe and desirable — same 204 for all users |
| Chain discovery (`GET /mining/`) | Yes | Purge after editing `chains.json`, or wait for TTL |

### Headers Passed Through

Varnish passes these headers by default (no extra config needed):

- **`Host`** — Required by HAProxy's `req.hdr(host)` ACLs and `map_dom`
- **`Origin`** — Required for CORS matching
- **`X-Admin-Secret`** — Required by the backend for admin authentication

If you have custom `unset req.http.*` rules in your VCL, verify these three headers are not stripped.

---

## Request Flow

```
1. Browser loads pools.securepayment.cc/index.html
2. Apache SSI injects: window.TIDEPOOL_GATEWAY = 'https://api.securepayment.cc/mining'
3. Frontend fetches https://api.securepayment.cc/mining/ → chains.json
4. Chain selector populated: [Bitcoin ▾] [Litecoin]
5. HTMX fires hx-get="/api/v1/stats"
6. configRequest handler rewrites → https://api.securepayment.cc/mining/bitcoin/v1/stats
7. HAProxy matches Host: api.securepayment.cc + path /mining/bitcoin/
8. HAProxy rewrites path → /api/v1/stats
9. HAProxy resolves SRV _http._tcp.bitcoin.api.securepayment.cc → api01:8001
10. Backend receives GET /api/v1/stats (zero code changes needed)
11. HAProxy adds CORS headers to response
```

---

## Troubleshooting

### CORS Errors in Browser Console

- Verify the frontend is loaded from `pools.securepayment.cc` (must match the `Access-Control-Allow-Origin` header)
- Check that `METH_OPTIONS` preflight returns 204 (not routed to a backend)
- Ensure the backend's own CORS headers are stripped (`http-response del-header Access-Control-Allow-Origin`)

### SRV Records Not Resolving

```sh
# Verify SRV record exists
drill SRV _http._tcp.bitcoin.api.securepayment.cc

# Check HAProxy's resolver is reaching DNS
haproxy -c -f /usr/local/etc/haproxy.conf

# Check HAProxy stats for backend server status
curl http://localhost:8181/
```

If `server-template` slots show as `DOWN`, the SRV record may not be resolving. Check the `nameserver` in the `resolvers dns` section.

### Path Rewrite Mismatches

Test the rewrite logic with curl:

```sh
# Should reach /api/v1/stats on the backend
curl -H "Host: api.securepayment.cc" http://localhost:8080/mining/bitcoin/v1/stats

# Direct access (no rewrite) for comparison
curl http://bitcoin.api.securepayment.cc/api/v1/stats
```

### Chain Discovery Not Working

```sh
# Should return chains.json content
curl -H "Host: api.securepayment.cc" http://localhost:8080/mining/
```

If this returns a 503 or routes to a backend, check the ACL order — `path_mining_root` must be evaluated before the per-chain `path_beg` ACLs.
