<?php
/**
 * TidePoolUI REST API Entry Point
 * Enchilada Framework 3.0
 */

require_once __DIR__ . '/../includes/bootstrap.inc.php';

use Enchilada\Config\IniConfig;

// --- Composition Root ---

/** @var IniConfig|null $SETTINGS */
global $SETTINGS;

$redis = tidepoolui_create_redis($SETTINGS);
$workerTtl = $SETTINGS ? $SETTINGS->getInt('cache', 'worker_ttl', 900) : 900;
$store = new ShareStore($redis, $workerTtl);
$geoip = new GeoIP($redis);
$adminSecret = $SETTINGS ? $SETTINGS->getString('admin', 'secret', '') : '';
$demoEnabled = $SETTINGS ? $SETTINGS->getBool('app', 'demo_enabled', false) : false;
$corsOrigins = $SETTINGS ? $SETTINGS->getString('api', 'cors_origins', '*') : '*';
$dateFormat = $SETTINGS ? $SETTINGS->getString('date', 'format', 'c') : 'c';
$chainId = $SETTINGS ? $SETTINGS->getString('bridge', 'chain_id', 'bitcoin') : 'bitcoin';

// BlockStore (MariaDB) — optional, only if mariadb section is configured
$blockStore = null;
if ($SETTINGS && $SETTINGS->has('mariadb', 'host')) {
    try {
        require_once __DIR__ . '/../includes/mariadb.inc.php';
        $pdo = tidepoolui_create_mariadb($SETTINGS);
        $blockStore = new BlockStore($pdo);
    } catch (PDOException $e) {
        // MariaDB not available — block endpoints will return empty results
    }
}

$poolStatsStore = new PoolStatsStore($redis, $blockStore);

// --- API Router ---

$api = new EnchiladaREST('/api/v1');

$api->enableCors([
    'origins' => $corsOrigins,
]);

// Helper: Check if request has valid admin secret
$isAdmin = function($req) use ($adminSecret): bool {
    if (empty($adminSecret)) {
        return false;
    }
    return $req->getHeader('X-Admin-Secret') === $adminSecret;
};

// Helper: Strip IP addresses from data array (non-admin privacy)
$stripAddresses = function(array $data, string $field = 'address'): array {
    foreach ($data as &$item) {
        unset($item[$field]);
    }
    return $data;
};

// Helper: Enrich data with GeoIP location (admin only)
$enrichAddresses = function(array $data, string $field = 'address') use ($geoip): array {
    foreach ($data as &$item) {
        if (isset($item[$field])) {
            $item['location'] = $geoip->getLocation($item[$field]) ?: 'Unknown';
        }
    }
    return $data;
};

// Health check
$api->get('/health', function($req, $res) {
    $res->success([
        'status' => 'ok',
        'version' => APPLICATION_VERSION,
        'timestamp' => date('c'),
    ]);
});

// Get app config (for frontend feature flags)
$api->get('/config', function($req, $res) use ($demoEnabled, $adminSecret) {
    $res->json([
        'demo_enabled' => $demoEnabled,
        'admin_enabled' => !empty($adminSecret),
    ]);
});

// Verify admin secret
$api->post('/admin/verify', function($req, $res) use ($isAdmin) {
    if ($isAdmin($req)) {
        $res->success(['admin' => true]);
    } else {
        $res->error('Invalid secret', 401);
    }
});

// Get recent shares
$api->get('/shares', function($req, $res) use ($store, $isAdmin, $stripAddresses, $enrichAddresses) {
    $limit = min(100, max(1, intval($req->getQueryParam('limit', 50))));
    $poolId = $req->getQueryParam('pool');
    if ($poolId === '' || $poolId === 'all') {
        $poolId = null;
    }
    
    $shares = $store->getRecentShares($limit, $poolId);
    
    if ($isAdmin($req)) {
        $shares = $enrichAddresses($shares);
    } else {
        $shares = $stripAddresses($shares);
    }
    
    // Return {shares: [...]} for Handlebars {{#each shares}}
    $res->json(['shares' => $shares]);
});

// Get pool statistics (supports ?pool=<pool_id> filter)
$api->get('/stats', function($req, $res) use ($store) {
    $poolId = $req->getQueryParam('pool');
    // Treat empty string or 'all' as null (aggregate)
    if ($poolId === '' || $poolId === 'all') {
        $poolId = null;
    }
    
    $stats = $store->getPoolStats($poolId);
    $hashrate = $store->getPoolHashrate($poolId);
    // Merge stats with hashrate data
    $res->json(array_merge($stats, $hashrate));
});

// Get list of known pools
$api->get('/pools', function($req, $res) use ($store) {
    $pools = $store->getPools();
    // Ensure we return an array, not a Redis set object
    $res->json(['pools' => array_values($pools)]);
});

// Get worker statistics
$api->get('/workers', function($req, $res) use ($store, $isAdmin, $stripAddresses, $enrichAddresses) {
    $poolId = $req->getQueryParam('pool');
    if ($poolId === '' || $poolId === 'all') {
        $poolId = null;
    }
    
    $workers = $store->getWorkerStats($poolId);
    
    // Add hashrate to each worker
    foreach ($workers as &$worker) {
        $workerName = $worker['workername'] ?? '';
        if ($workerName) {
            $hashrate = $store->getWorkerHashrate($workerName);
            $worker = array_merge($worker, $hashrate);
        }
    }
    
    if ($isAdmin($req)) {
        $workers = $enrichAddresses($workers);
    } else {
        $workers = $stripAddresses($workers);
    }
    
    // Return {workers: [...]} for Handlebars {{#each workers}}
    $res->json(['workers' => $workers]);
});

// Add demo data (for testing without Kafka)
$api->post('/demo', function($req, $res) use ($store, $demoEnabled, $dateFormat) {
    // Check if demo mode is enabled
    if (!$demoEnabled) {
        $res->error('Demo mode is disabled', 403);
        return;
    }
    
    $workers = ['worker1.rig1', 'worker1.rig2', 'miner2.antminer', 'pool.test'];
    // Mix of private and public IPs for testing GeoIP
    $addresses = [
        '192.168.1.100',  // Private
        '10.0.0.50',      // Private
        '8.8.8.8',        // Google DNS (Mountain View, CA)
        '1.1.1.1',        // Cloudflare (San Francisco, CA)
        '208.67.222.222', // OpenDNS (San Francisco, CA)
    ];
    $now = time();
    
    for ($i = 0; $i < 10; $i++) {
        $worker = $workers[array_rand($workers)];
        $username = explode('.', $worker)[0];
        $valid = rand(0, 100) > 5; // 95% valid
        
        $store->addShare([
            'workinfoid' => rand(100000, 999999),
            'clientid' => rand(1, 100),
            'enonce1' => bin2hex(random_bytes(4)),
            'nonce2' => bin2hex(random_bytes(8)),
            'nonce' => bin2hex(random_bytes(4)),
            'ntime' => dechex($now - rand(0, 60)),
            'diff' => 1.0,
            'sdiff' => round(rand(100, 5000) / 1000, 3),
            'hash' => '0000000000000' . bin2hex(random_bytes(20)),
            'result' => $valid,
            'errn' => $valid ? 0 : 21,
            'createdate' => date($dateFormat, $now - rand(0, 60)),
            'workername' => $worker,
            'username' => $username,
            'address' => $addresses[array_rand($addresses)],
            'agent' => 'cgminer/4.11.1',
        ]);
    }
    
    $res->success(['message' => 'Added 10 demo shares']);
});

// --- Phase 3: Pool Stats Endpoints ---

// Get network stats (from TidePoolBridge → Redis)
$api->get('/network', function($req, $res) use ($poolStatsStore, $chainId) {
    $chain = $req->getQueryParam('chain', $chainId);
    $res->json($poolStatsStore->getNetworkStats($chain));
});

// Get solved blocks (from MariaDB)
$api->get('/blocks', function($req, $res) use ($blockStore) {
    if (!$blockStore) {
        $res->json(['blocks' => [], 'total' => 0]);
        return;
    }
    
    $limit = min(100, max(1, intval($req->getQueryParam('limit', 50))));
    $user = $req->getQueryParam('user');
    $poolId = $req->getQueryParam('pool');
    if ($user === '') { $user = null; }
    if ($poolId === '' || $poolId === 'all') { $poolId = null; }
    
    $blocks = $blockStore->getBlocks($limit, $poolId, $user);
    $total = $blockStore->getBlockCount($poolId, $user);
    
    $res->json([
        'blocks' => $blocks,
        'total' => $total,
    ]);
});

// Get per-user stats (Redis workers + MariaDB blocks)
$api->get('/user/{address}', function($req, $res) use ($poolStatsStore, $chainId) {
    $address = $req->getRouteParam('address');
    $chain = $req->getQueryParam('chain', $chainId);
    
    $stats = $poolStatsStore->getUserStats($address, $chain);
    if ($stats === null) {
        $res->error('User not found or no activity', 404);
        return;
    }
    
    $res->json($stats);
});

// Run the API
$api->run();
