<?php
/**
 * BlockStore - MariaDB-backed block storage
 * 
 * Stores solved blocks consumed from Kafka for display in the pool stats UI.
 * Uses PDO for MariaDB access.
 */
class BlockStore
{
    private PDO $db;
    
    /**
     * @param PDO $db Connected PDO instance
     */
    public function __construct(PDO $db)
    {
        $this->db = $db;
        $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    }
    
    /**
     * Add a solved block from Kafka
     *
     * @param array $block Block data from Kafka message
     * @return bool True if inserted, false if duplicate (hash already exists)
     */
    public function addBlock(array $block): bool
    {
        $sql = "INSERT IGNORE INTO blocks (height, hash, workername, username, difficulty, pool_id, confirmed)
                VALUES (:height, :hash, :workername, :username, :difficulty, :pool_id, :confirmed)";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':height'     => (int)($block['height'] ?? 0),
            ':hash'       => $block['hash'] ?? '',
            ':workername' => $block['workername'] ?? 'unknown',
            ':username'   => $block['username'] ?? $this->extractUsername($block['workername'] ?? ''),
            ':difficulty' => (float)($block['diff'] ?? 0),
            ':pool_id'    => $block['pool_id'] ?? 'default',
            ':confirmed'  => (bool)($block['confirmed'] ?? false),
        ]);
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Get recent blocks, optionally filtered
     *
     * @param int         $limit  Max blocks to return
     * @param string|null $poolId Filter by pool
     * @param string|null $user   Filter by username
     * @return array
     */
    public function getBlocks(int $limit = 100, ?string $poolId = null, ?string $user = null): array
    {
        $where = [];
        $params = [];
        
        if ($poolId !== null) {
            $where[] = 'pool_id = :pool_id';
            $params[':pool_id'] = $poolId;
        }
        if ($user !== null) {
            $where[] = 'username = :username';
            $params[':username'] = $user;
        }
        
        $sql = "SELECT * FROM blocks";
        if ($where) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        $sql .= " ORDER BY height DESC LIMIT :limit";
        
        $stmt = $this->db->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Get block count, optionally filtered
     */
    public function getBlockCount(?string $poolId = null, ?string $user = null): int
    {
        $where = [];
        $params = [];
        
        if ($poolId !== null) {
            $where[] = 'pool_id = :pool_id';
            $params[':pool_id'] = $poolId;
        }
        if ($user !== null) {
            $where[] = 'username = :username';
            $params[':username'] = $user;
        }
        
        $sql = "SELECT COUNT(*) FROM blocks";
        if ($where) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return (int)$stmt->fetchColumn();
    }
    
    /**
     * Update block confirmation status
     */
    public function confirmBlock(string $hash): bool
    {
        $stmt = $this->db->prepare("UPDATE blocks SET confirmed = TRUE WHERE hash = :hash");
        $stmt->execute([':hash' => $hash]);
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Extract username from workername (format: "username.workername")
     */
    private function extractUsername(string $workername): string
    {
        $parts = explode('.', $workername, 2);
        return $parts[0] ?: 'unknown';
    }
}
