<?php
/**
 * PoolStatsStore - Reads network and pool stats from Redis
 * 
 * Consumes data written by TidePoolBridge:
 *   - chain:{chain_id}:network   — bitcoind network stats (height, difficulty, nethash)
 *   - chain:{chain_id}:poolstats — SeaTidePool pool stats (workers, sps, dsps)
 * 
 * Also reads block data from MariaDB via BlockStore.
 */
class PoolStatsStore
{
    private Redis $redis;
    private ?BlockStore $blockStore;
    
    // Hashrate calculation (matches SeaTidePool / ShareStore)
    private const NONCES = 4294967296; // 2^32
    
    /**
     * @param Redis      $redis      Connected Redis instance
     * @param BlockStore|null $blockStore Optional MariaDB block store
     */
    public function __construct(Redis $redis, ?BlockStore $blockStore = null)
    {
        $this->redis = $redis;
        $this->blockStore = $blockStore;
    }
    
    /**
     * Get network stats for a chain (from TidePoolBridge → Redis)
     *
     * @param string $chainId Chain identifier (e.g., "bitcoin")
     * @return array Network stats or defaults if unavailable
     */
    public function getNetworkStats(string $chainId = 'bitcoin'): array
    {
        $data = $this->redis->hGetAll("chain:{$chainId}:network");
        
        if (!$data) {
            return [
                'height' => 0,
                'difficulty' => 0,
                'nethash_raw' => 0,
                'nethash' => '0.00 H/s',
                'best_block' => '',
                'chain' => $chainId,
                'updated_at' => null,
                'stale' => true,
            ];
        }
        
        // Check staleness (no update in 10 minutes = stale)
        $updatedAt = $data['updated_at'] ?? null;
        $stale = true;
        if ($updatedAt) {
            $age = time() - strtotime($updatedAt);
            $stale = $age > 600;
        }
        
        return [
            'height' => (int)($data['height'] ?? 0),
            'difficulty' => (float)($data['difficulty'] ?? 0),
            'nethash_raw' => (float)($data['nethash_raw'] ?? 0),
            'nethash' => $data['nethash'] ?? '0.00 H/s',
            'best_block' => $data['best_block'] ?? '',
            'chain' => $data['chain'] ?? $chainId,
            'updated_at' => $updatedAt,
            'stale' => $stale,
        ];
    }
    
    /**
     * Get pool stats for a chain (from TidePoolBridge → Redis)
     *
     * @param string $chainId Chain identifier
     * @return array Pool stats or defaults
     */
    public function getPoolStats(string $chainId = 'bitcoin'): array
    {
        $data = $this->redis->hGetAll("chain:{$chainId}:poolstats");
        
        if (!$data) {
            return [
                'workers' => 0,
                'users' => 0,
                'accepted' => 0,
                'rejected' => 0,
                'pool_hashrate' => '0.00 H/s',
                'pool_hashrate_raw' => 0,
                'sps1' => 0,
                'sps5' => 0,
                'sps15' => 0,
                'sps60' => 0,
                'updated_at' => null,
                'stale' => true,
            ];
        }
        
        // Calculate pool hashrate from dsps (diff shares per second)
        $dsps1 = (float)($data['dsps1'] ?? 0);
        $dsps5 = (float)($data['dsps5'] ?? 0);
        $dsps15 = (float)($data['dsps15'] ?? 0);
        $dsps60 = (float)($data['dsps60'] ?? 0);
        
        $updatedAt = $data['updated_at'] ?? null;
        $stale = true;
        if ($updatedAt) {
            $age = time() - strtotime($updatedAt);
            $stale = $age > 120; // Pool stats are every 10-15s, stale after 2 min
        }
        
        return [
            'workers' => (int)($data['workers'] ?? 0),
            'users' => (int)($data['users'] ?? 0),
            'accepted' => (int)($data['accepted'] ?? 0),
            'rejected' => (int)($data['rejected'] ?? 0),
            'pool_hashrate' => $this->formatHashrate($dsps5 * self::NONCES),
            'pool_hashrate_raw' => $dsps5 * self::NONCES,
            'hashrate_1m' => $this->formatHashrate($dsps1 * self::NONCES),
            'hashrate_5m' => $this->formatHashrate($dsps5 * self::NONCES),
            'hashrate_15m' => $this->formatHashrate($dsps15 * self::NONCES),
            'hashrate_1h' => $this->formatHashrate($dsps60 * self::NONCES),
            'hashrate_1m_raw' => $dsps1 * self::NONCES,
            'hashrate_5m_raw' => $dsps5 * self::NONCES,
            'hashrate_15m_raw' => $dsps15 * self::NONCES,
            'hashrate_1h_raw' => $dsps60 * self::NONCES,
            'sps1' => round((float)($data['sps1'] ?? $data['SPS1m'] ?? 0), 4),
            'sps5' => round((float)($data['sps5'] ?? $data['SPS5m'] ?? 0), 4),
            'sps15' => round((float)($data['sps15'] ?? $data['SPS15m'] ?? 0), 4),
            'sps60' => round((float)($data['sps60'] ?? $data['SPS1h'] ?? 0), 4),
            'bestshare' => (float)($data['bestshare'] ?? 0),
            'network_diff' => (float)($data['network_diff'] ?? 0),
            'updated_at' => $updatedAt,
            'stale' => $stale,
        ];
    }
    
    /**
     * Get per-user aggregate stats
     *
     * @param string $username Username (bitcoin address)
     * @param string $chainId  Chain identifier
     * @return array|null User stats or null if not found
     */
    public function getUserStats(string $username, string $chainId = 'bitcoin'): ?array
    {
        $result = [
            'username' => $username,
            'chain_id' => $chainId,
        ];
        
        // Block stats from MariaDB
        if ($this->blockStore) {
            $result['blocks_found'] = $this->blockStore->getBlockCount(null, $username);
            $result['blocks_confirmed'] = $this->blockStore->getBlockCount(null, $username);
            $result['recent_blocks'] = $this->blockStore->getBlocks(10, null, $username);
        } else {
            $result['blocks_found'] = 0;
            $result['blocks_confirmed'] = 0;
            $result['recent_blocks'] = [];
        }
        
        // Worker info from Redis (ShareStore keys)
        $workers = $this->redis->sMembers('tidepoolui:workers') ?: [];
        $userWorkers = [];
        $totalShares = 0;
        $totalValid = 0;
        
        foreach ($workers as $worker) {
            $workerData = $this->redis->hGetAll('tidepoolui:workers:' . $worker);
            if ($workerData && ($workerData['username'] ?? '') === $username) {
                $userWorkers[] = $workerData['workername'] ?? $worker;
                $totalShares += (int)($workerData['shares'] ?? 0);
                $totalValid += (int)($workerData['valid'] ?? 0);
            }
        }
        
        $result['workers_active'] = count($userWorkers);
        $result['worker_names'] = $userWorkers;
        $result['total_shares'] = $totalShares;
        $result['valid_shares'] = $totalValid;
        
        // Only return if user has any activity
        if ($totalShares === 0 && $result['blocks_found'] === 0) {
            return null;
        }
        
        return $result;
    }
    
    /**
     * Format hashrate as human-readable string
     */
    private function formatHashrate(float $hashrate): string
    {
        $units = ['H/s', 'KH/s', 'MH/s', 'GH/s', 'TH/s', 'PH/s', 'EH/s'];
        $unit = 0;
        while ($hashrate >= 1000 && $unit < count($units) - 1) {
            $hashrate /= 1000;
            $unit++;
        }
        return sprintf('%.2f %s', $hashrate, $units[$unit]);
    }
}
