<?php
/**
 * ShareStore - Redis-backed share storage with statistics
 * 
 * Stores recent shares consumed from Kafka for display in the UI.
 * Uses Redis lists for the ring buffer and hashes for stats.
 */
class ShareStore
{
    private Redis $redis;
    private int $maxShares = 1000;
    private int $workerTtl = 900; // 15 minutes default
    
    // Redis keys
    private const KEY_SHARES = 'tidepoolui:shares';
    private const KEY_POOL_STATS = 'tidepoolui:pool_stats';
    private const KEY_WORKERS = 'tidepoolui:workers';
    private const KEY_HASHRATE = 'tidepoolui:hashrate';
    private const KEY_POOLS = 'tidepoolui:pools'; // Set of known pool_ids
    
    // Hashrate calculation constants (matches SeaTidePool)
    private const NONCES = 4294967296; // 2^32
    private const MIN1 = 60;
    private const MIN5 = 300;
    private const MIN15 = 900;
    private const HOUR = 3600;
    
    /**
     * @param Redis $redis     Connected Redis instance
     * @param int   $workerTtl Seconds before worker considered offline
     */
    public function __construct(Redis $redis, int $workerTtl = 900)
    {
        $this->redis = $redis;
        $this->workerTtl = $workerTtl;
    }
    
    /**
     * Test Redis connection
     * @throws RedisException if connection is dead
     */
    public function ping(): bool
    {
        return $this->redis->ping() === true || $this->redis->ping() === '+PONG';
    }
    
    /**
     * Add a share to the store
     */
    public function addShare(array $share): void
    {
        // Add to list (ring buffer)
        $this->redis->rPush(self::KEY_SHARES, json_encode($share));
        $this->redis->lTrim(self::KEY_SHARES, -$this->maxShares, -1);
        
        // Get pool_id (default to 'default' if not set)
        $poolId = $share['pool_id'] ?? 'default';
        
        // Track this pool
        $this->redis->sAdd(self::KEY_POOLS, $poolId);
        
        // Update aggregate pool stats (all pools combined)
        $this->updatePoolStats(self::KEY_POOL_STATS, $share);
        
        // Update per-pool stats
        $this->updatePoolStats(self::KEY_POOL_STATS . ':' . $poolId, $share);
        
        // Update worker stats
        $worker = $share['workername'] ?? 'unknown';
        $workerKey = self::KEY_WORKERS . ':' . $worker;
        
        $isNew = !$this->redis->exists($workerKey);
        if ($isNew) {
            $this->redis->hMSet($workerKey, [
                'workername' => $worker,
                'username' => $share['username'] ?? 'unknown',
                'shares' => 0,
                'valid' => 0,
                'invalid' => 0,
                'last_diff' => 0,
                'total_diff' => 0,
                'address' => $share['address'] ?? '',
                'agent' => $share['agent'] ?? '',
            ]);
            $this->redis->sAdd(self::KEY_WORKERS, $worker);
            $this->redis->hIncrBy(self::KEY_POOL_STATS, 'workers_seen', 1);
        }
        
        // Track workers per pool (use set to track unique workers per pool)
        $poolWorkersKey = self::KEY_POOL_STATS . ':' . $poolId . ':workers';
        $isNewToPool = $this->redis->sAdd($poolWorkersKey, $worker);
        if ($isNewToPool) {
            $this->redis->hIncrBy(self::KEY_POOL_STATS . ':' . $poolId, 'workers_seen', 1);
        }
        
        $this->redis->hIncrBy($workerKey, 'shares', 1);
        if ($share['result'] ?? false) {
            $this->redis->hIncrBy($workerKey, 'valid', 1);
        } else {
            $this->redis->hIncrBy($workerKey, 'invalid', 1);
        }
        $this->redis->hSet($workerKey, 'last_diff', $share['sdiff'] ?? 0);
        $this->redis->hIncrByFloat($workerKey, 'total_diff', $share['sdiff'] ?? 0);
        $this->redis->hSet($workerKey, 'last_seen', $share['createdate'] ?? date('Y-m-d H:i:s'));
        
        // Refresh TTL on worker key (keeps active workers alive)
        $this->redis->expire($workerKey, $this->workerTtl);
        
        // Update hashrate metrics (EWMA decay)
        // Use 'diff' (pool difficulty) not 'sdiff' (actual share difficulty)
        // This matches SeaTidePool's calculation in stratifier.c
        $this->updateHashrateMetrics($share['diff'] ?? 1, $worker, $poolId);
    }
    
    /**
     * Update pool statistics for a specific stats key
     */
    private function updatePoolStats(string $key, array $share): void
    {
        $this->redis->hIncrBy($key, 'total_shares', 1);
        if ($share['result'] ?? false) {
            $this->redis->hIncrBy($key, 'valid_shares', 1);
        } else {
            $this->redis->hIncrBy($key, 'invalid_shares', 1);
        }
        $this->redis->hSet($key, 'last_share_time', $share['createdate'] ?? date('Y-m-d H:i:s'));
    }
    
    /**
     * Exponentially decaying average (matches SeaTidePool's decay_time)
     */
    private function decayTime(float $current, float $diff, float $tdiff, float $interval): float
    {
        if ($tdiff <= 0) {
            return $current;
        }
        $dexp = $tdiff / $interval;
        if ($dexp > 36) {
            $dexp = 36;
        }
        $fprop = 1.0 - 1 / exp($dexp);
        $ftotal = 1.0 + $fprop;
        $result = ($current + ($diff / $tdiff * $fprop)) / $ftotal;
        // Prevent underflow
        if ($result < 2E-16) {
            $result = 0;
        }
        return $result;
    }
    
    /**
     * Update hashrate metrics using EWMA
     */
    private function updateHashrateMetrics(float $shareDiff, string $worker, string $poolId = 'default'): void
    {
        $now = microtime(true);
        
        // Update aggregate pool-wide hashrate (all pools)
        $this->updateEntityHashrate(self::KEY_HASHRATE . ':pool', $shareDiff, $now);
        
        // Update per-pool hashrate
        $this->updateEntityHashrate(self::KEY_HASHRATE . ':pool:' . $poolId, $shareDiff, $now);
        
        // Update per-worker hashrate
        $this->updateEntityHashrate(self::KEY_HASHRATE . ':worker:' . $worker, $shareDiff, $now);
    }
    
    /**
     * Update hashrate for a specific entity (pool or worker)
     */
    private function updateEntityHashrate(string $key, float $shareDiff, float $now): void
    {
        $data = $this->redis->hGetAll($key);
        $lastUpdate = (float)($data['last_update'] ?? $now);
        $tdiff = $now - $lastUpdate;
        
        // Get current dsps values
        $dsps1 = (float)($data['dsps1'] ?? 0);
        $dsps5 = (float)($data['dsps5'] ?? 0);
        $dsps15 = (float)($data['dsps15'] ?? 0);
        $dsps60 = (float)($data['dsps60'] ?? 0);
        
        // Apply decay with new share
        $dsps1 = $this->decayTime($dsps1, $shareDiff, $tdiff, self::MIN1);
        $dsps5 = $this->decayTime($dsps5, $shareDiff, $tdiff, self::MIN5);
        $dsps15 = $this->decayTime($dsps15, $shareDiff, $tdiff, self::MIN15);
        $dsps60 = $this->decayTime($dsps60, $shareDiff, $tdiff, self::HOUR);
        
        // Store updated values
        $this->redis->hMSet($key, [
            'dsps1' => $dsps1,
            'dsps5' => $dsps5,
            'dsps15' => $dsps15,
            'dsps60' => $dsps60,
            'last_update' => $now,
        ]);
        
        // Set TTL on worker hashrate keys
        if (strpos($key, ':worker:') !== false) {
            $this->redis->expire($key, $this->workerTtl);
        }
    }
    
    /**
     * Get pool hashrate statistics (aggregate or filtered by pool_id)
     */
    public function getPoolHashrate(?string $poolId = null): array
    {
        $key = $poolId 
            ? self::KEY_HASHRATE . ':pool:' . $poolId 
            : self::KEY_HASHRATE . ':pool';
        $data = $this->redis->hGetAll($key);
        if (!$data) {
            return $this->formatHashrate(0, 0, 0, 0);
        }
        
        // Decay values to current time (in case no recent shares)
        $now = microtime(true);
        $lastUpdate = (float)($data['last_update'] ?? $now);
        $tdiff = $now - $lastUpdate;
        
        $dsps1 = $this->decayTime((float)($data['dsps1'] ?? 0), 0, $tdiff, self::MIN1);
        $dsps5 = $this->decayTime((float)($data['dsps5'] ?? 0), 0, $tdiff, self::MIN5);
        $dsps15 = $this->decayTime((float)($data['dsps15'] ?? 0), 0, $tdiff, self::MIN15);
        $dsps60 = $this->decayTime((float)($data['dsps60'] ?? 0), 0, $tdiff, self::HOUR);
        
        return $this->formatHashrate($dsps1, $dsps5, $dsps15, $dsps60);
    }
    
    /**
     * Get worker hashrate
     */
    public function getWorkerHashrate(string $worker): array
    {
        $data = $this->redis->hGetAll(self::KEY_HASHRATE . ':worker:' . $worker);
        if (!$data) {
            return $this->formatHashrate(0, 0, 0, 0);
        }
        
        $now = microtime(true);
        $lastUpdate = (float)($data['last_update'] ?? $now);
        $tdiff = $now - $lastUpdate;
        
        $dsps1 = $this->decayTime((float)($data['dsps1'] ?? 0), 0, $tdiff, self::MIN1);
        $dsps5 = $this->decayTime((float)($data['dsps5'] ?? 0), 0, $tdiff, self::MIN5);
        $dsps15 = $this->decayTime((float)($data['dsps15'] ?? 0), 0, $tdiff, self::MIN15);
        $dsps60 = $this->decayTime((float)($data['dsps60'] ?? 0), 0, $tdiff, self::HOUR);
        
        return $this->formatHashrate($dsps1, $dsps5, $dsps15, $dsps60);
    }
    
    /**
     * Format hashrate values with human-readable strings
     */
    private function formatHashrate(float $dsps1, float $dsps5, float $dsps15, float $dsps60): array
    {
        return [
            'hashrate_1m' => $dsps1 * self::NONCES,
            'hashrate_5m' => $dsps5 * self::NONCES,
            'hashrate_15m' => $dsps15 * self::NONCES,
            'hashrate_1h' => $dsps60 * self::NONCES,
            'hashrate_1m_human' => $this->formatHashrateHuman($dsps1 * self::NONCES),
            'hashrate_5m_human' => $this->formatHashrateHuman($dsps5 * self::NONCES),
            'hashrate_15m_human' => $this->formatHashrateHuman($dsps15 * self::NONCES),
            'hashrate_1h_human' => $this->formatHashrateHuman($dsps60 * self::NONCES),
        ];
    }
    
    /**
     * Format hashrate as human-readable string (e.g., "1.23 TH/s")
     */
    private function formatHashrateHuman(float $hashrate): string
    {
        $units = ['H/s', 'KH/s', 'MH/s', 'GH/s', 'TH/s', 'PH/s', 'EH/s'];
        $unit = 0;
        while ($hashrate >= 1000 && $unit < count($units) - 1) {
            $hashrate /= 1000;
            $unit++;
        }
        return sprintf('%.2f %s', $hashrate, $units[$unit]);
    }
    
    /**
     * Get recent shares, optionally filtered by pool
     */
    public function getRecentShares(int $limit = 50, ?string $poolId = null): array
    {
        $shares = $this->redis->lRange(self::KEY_SHARES, -$limit * 2, -1); // Get extra for filtering
        if (!is_array($shares)) { return []; }
        $decoded = array_map(fn($s) => json_decode($s, true), $shares);
        
        // Filter by pool if specified
        if ($poolId) {
            $decoded = array_filter($decoded, fn($s) => ($s['pool_id'] ?? 'default') === $poolId);
        }
        
        $decoded = array_slice($decoded, -$limit); // Limit after filtering
        return array_reverse($decoded); // Most recent first
    }
    
    /**
     * Get worker statistics, optionally filtered by pool
     */
    public function getWorkerStats(?string $poolId = null): array
    {
        // Get workers - either all or filtered by pool
        if ($poolId) {
            $poolWorkersKey = self::KEY_POOL_STATS . ':' . $poolId . ':workers';
            $workers = $this->redis->sMembers($poolWorkersKey) ?: [];
        } else {
            $workers = $this->redis->sMembers(self::KEY_WORKERS) ?: [];
        }
        
        $stats = [];
        $expiredWorkers = [];
        
        foreach ($workers as $worker) {
            $workerKey = self::KEY_WORKERS . ':' . $worker;
            $data = $this->redis->hGetAll($workerKey);
            if ($data) {
                $stats[] = $data;
            } else {
                // Worker key expired, mark for removal from set
                $expiredWorkers[] = $worker;
            }
        }
        
        // Clean up expired workers from the set
        foreach ($expiredWorkers as $worker) {
            $this->redis->sRem(self::KEY_WORKERS, $worker);
        }
        
        return $stats;
    }
    
    /**
     * Get pool statistics (aggregate or filtered by pool_id)
     */
    public function getPoolStats(?string $poolId = null): array
    {
        $key = $poolId ? self::KEY_POOL_STATS . ':' . $poolId : self::KEY_POOL_STATS;
        $stats = $this->redis->hGetAll($key);
        if (!$stats) {
            $stats = [
                'total_shares' => 0,
                'valid_shares' => 0,
                'invalid_shares' => 0,
                'workers_seen' => 0,
                'last_share_time' => null,
            ];
        }
        
        $stats['pool_id'] = $poolId ?: 'all';
        $stats['workers_active'] = $this->getActiveWorkerCount($poolId);
        $stats['shares_per_minute'] = $this->calculateSharesPerMinute($poolId);
        $total = (int)($stats['total_shares'] ?? 0);
        $valid = (int)($stats['valid_shares'] ?? 0);
        $stats['acceptance_rate'] = $total > 0 
            ? round(($valid / $total) * 100, 2) 
            : 0;
        return $stats;
    }
    
    /**
     * Count currently active workers (those with live Redis keys)
     */
    public function getActiveWorkerCount(?string $poolId = null): int
    {
        if ($poolId) {
            $setKey = self::KEY_POOL_STATS . ':' . $poolId . ':workers';
        } else {
            $setKey = self::KEY_WORKERS;
        }
        
        $workers = $this->redis->sMembers($setKey) ?: [];
        $active = 0;
        
        foreach ($workers as $worker) {
            if ($this->redis->exists(self::KEY_WORKERS . ':' . $worker)) {
                $active++;
            } else {
                $this->redis->sRem($setKey, $worker);
            }
        }
        
        return $active;
    }
    
    /**
     * Get list of known pool IDs
     */
    public function getPools(): array
    {
        $pools = $this->redis->sMembers(self::KEY_POOLS);
        return $pools ?: [];
    }
    
    /**
     * Calculate shares per minute from recent data
     */
    private function calculateSharesPerMinute(?string $poolId = null): float
    {
        $shares = $this->redis->lRange(self::KEY_SHARES, -100, -1);
        if (!is_array($shares) || count($shares) < 2) {
            return 0;
        }
        
        // Filter by pool_id if specified
        $decoded = array_map(fn($s) => json_decode($s, true), $shares);
        if ($poolId) {
            $decoded = array_filter($decoded, fn($s) => ($s['pool_id'] ?? 'default') === $poolId);
            $decoded = array_values($decoded);
        }
        
        if (count($decoded) < 2) {
            return 0;
        }
        
        $first = $decoded[0];
        $last = $decoded[count($decoded) - 1];
        $firstTime = strtotime($first['createdate'] ?? 'now');
        $lastTime = strtotime($last['createdate'] ?? 'now');
        $minutes = max(1, ($lastTime - $firstTime) / 60);
        
        return round(count($decoded) / $minutes, 2);
    }
    
    /**
     * Clear all data (for testing)
     */
    public function clear(): void
    {
        $workers = $this->redis->sMembers(self::KEY_WORKERS) ?: [];
        foreach ($workers as $worker) {
            $this->redis->del(self::KEY_WORKERS . ':' . $worker);
        }
        $this->redis->del(self::KEY_SHARES);
        $this->redis->del(self::KEY_POOL_STATS);
        $this->redis->del(self::KEY_WORKERS);
    }
}
