# TidePoolUI Architecture

This document describes the architecture and data flow of TidePoolUI, a real-time monitoring dashboard for SeaTidePool mining pools.

## System Overview

TidePoolUI is a PHP-based web application that displays real-time mining statistics from a SeaTidePool mining pool. It uses Kafka for message streaming and Redis for data storage.

```mermaid
graph LR
    A[Mining Hardware] -->|Stratum| B[SeaTidePool]
    B -->|Publish| C[Kafka]
    C -->|Consume| D[PHP Consumer]
    D -->|Store| E[Redis]
    E -->|Query| F[PHP API]
    F -->|JSON| G[HTMX Frontend]
```

## Components

### 1. SeaTidePool (External)

The mining pool server that:
- Accepts stratum connections from miners
- Validates submitted shares
- Publishes share events to Kafka topic `tidepool.dev.shares`

### 2. Kafka Message Broker (External)

Apache Kafka handles the message stream:
- **Topic**: `tidepool.dev.shares` (or `tidepool.prod.shares` in production)
- **Message Format**: JSON with share data
- **Retention**: Configurable, typically 7 days

Example share message:
```json
{
    "workinfoid": 12345,
    "clientid": 1,
    "diff": 1.0,
    "sdiff": 2.5,
    "result": true,
    "username": "miner1",
    "workername": "miner1.rig1",
    "address": "192.168.1.100",
    "agent": "bfgminer/5.5.0"
}
```

### 3. Kafka Consumer (`backend/bin/share-consumer.php`)

A long-running PHP process that:
- Connects to Kafka using the `rdkafka` extension
- Subscribes to the shares topic
- Processes each share message
- Stores data in Redis via the `ShareStore` class

**Running the consumer:**
```sh
./backend/bin/share-consumer.php
```

**Environment variables:**
| Variable | Default | Description |
|----------|---------|-------------|
| `KAFKA_BROKERS` | `localhost:9092` | Kafka bootstrap servers |
| `KAFKA_TOPIC` | `tidepool.dev.shares` | Topic to consume |
| `KAFKA_GROUP` | `tidepoolui-consumer` | Consumer group ID |

### 4. Redis Data Store

Redis stores three types of data:

| Key | Type | Description |
|-----|------|-------------|
| `tidepoolui:shares` | List | Recent shares (capped at 1000) |
| `tidepoolui:workers` | Set | Active worker names |
| `tidepoolui:worker:{name}` | Hash | Per-worker statistics |
| `tidepoolui:pool_stats` | Hash | Aggregate pool statistics |

### 5. PHP REST API (`backend/api/index.php`)

Built on the Enchilada micro-framework, provides:

| Endpoint | Method | Description |
|----------|--------|-------------|
| `/api/v1/shares` | GET | Recent shares list |
| `/api/v1/workers` | GET | Worker statistics |
| `/api/v1/stats` | GET | Pool statistics |
| `/api/v1/health` | GET | Health check |
| `/api/v1/demo` | POST | Add demo data |

### 6. Frontend (`frontend/index.html`)

Single-page application using:
- **HTMX**: Declarative AJAX requests
- **Handlebars**: Client-side templating
- **Pico CSS**: Lightweight styling

## Production API Gateway

In production, the frontend and API backends are on different domains. An HAProxy API gateway handles routing, CORS, and path translation.

```mermaid
graph LR
    A[Browser] -->|pool.securepayment.cc| B[Apache + SSI]
    B -->|Injects API_GATEWAY| A
    A -->|api.securepayment.cc/mining/bitcoin/v1/...| C[HAProxy]
    C -->|Path rewrite: /api/v1/...| D[Backend Apache]
    D --> E[PHP API]
    E --> F[Redis]
    C -->|SRV discovery| D
```

- **Apache SSI** injects the gateway base URL (`API_GATEWAY`) into the frontend at serve-time
- **Chain selector** fetches available chains from `GET /mining/` (static JSON served by HAProxy)
- **HTMX `configRequest`** rewrites all `/api/v1/...` URLs to `{gateway}/{chain}/v1/...` at runtime
- **HAProxy** strips the chain prefix and forwards `/api/v1/...` to the backend (zero backend changes)
- **SRV records** provide automatic server discovery — no hardcoded IPs/ports

See [HAPROXY_GATEWAY.md](HAPROXY_GATEWAY.md) for the complete configuration and setup guide.

## Data Flow

### Share Submission Flow

1. Miner submits share via stratum protocol
2. SeaTidePool validates share
3. SeaTidePool publishes to Kafka
4. PHP consumer receives message
5. Consumer calls `ShareStore::addShare()`
6. ShareStore updates Redis

### UI Display Flow

1. Page loads, HTMX triggers initial requests
2. API queries Redis via ShareStore
3. JSON response returned
4. Handlebars renders templates
5. HTMX polls every 3-10 seconds for updates

## Class Diagram

```mermaid
classDiagram
    class ShareStore {
        -Redis redis
        -int workerTtl
        +__construct(Redis redis, int workerTtl)
        +addShare(array share) void
        +getRecentShares(int limit) array
        +getWorkerStats() array
        +getPoolStats() array
        +getPoolHashrate() array
        +getWorkerHashrate(string worker) array
        +clear() void
    }
    
    class GeoIP {
        -Redis redis
        +__construct(Redis redis)
        +maskIP(string ip) string
        +getLocation(string ip) string
        +isPrivateIP(string ip) bool
    }
    
    class IniConfig {
        +load(string filename) IniConfig
        +getString(string section, string key) string
        +getInt(string section, string key) int
        +getBool(string section, string key) bool
    }
    
    class EnchiladaREST {
        +get(string path, callable handler)
        +post(string path, callable handler)
        +run()
    }
    
    ShareStore --> Redis : injected
    GeoIP --> Redis : injected
    EnchiladaREST --> EnchiladaResponse
```

Dependencies are wired at the **composition root** (API `index.php` and consumer `share-consumer.php`)
rather than via singletons or globals. Redis is created from `IniConfig` settings and injected into
`ShareStore` and `GeoIP`.

## File Structure

```
TidePoolUI/
├── backend/
│   ├── api/
│   │   └── index.php              # REST API composition root & routes
│   ├── bin/
│   │   └── share-consumer.php     # Kafka consumer daemon
│   ├── classes/                   # Application classes
│   │   ├── sharestore.class.php
│   │   └── geoip.class.php
│   ├── config/
│   │   ├── settings.ini.sample    # INI config template
│   │   └── local.conf.php.sample  # Pre-bootstrap overrides
│   ├── includes/
│   │   ├── bootstrap.inc.php      # Enchilada 3.0 bootstrap
│   │   ├── settings.inc.php       # Settings loader component
│   │   └── redis.inc.php          # Redis connection factory
│   ├── libraries/                 # Enchilada framework libraries
│   │   ├── Enchilada/
│   │   │   ├── Config/IniConfig.php
│   │   │   └── Daemon/DaemonBehavior.trait.php
│   │   └── REST/
│   │       ├── EnchiladaREST.class.php
│   │       ├── EnchiladaRequest.class.php
│   │       └── EnchiladaResponse.class.php
│   └── system/
│       ├── app.conf.php           # Application constants
│       └── autoload.inc.php       # Enchilada 3.0 autoloader
├── frontend/
│   ├── index.html                 # Main UI
│   ├── css/app.css
│   └── js/app.js
├── doc/                           # Documentation
├── etc/                           # FreeBSD RC scripts
├── ports/                         # FreeBSD port definitions
└── dev.sh                         # Development helper script
```

## Security Considerations

- Redis should not be exposed externally
- Kafka ACLs should restrict topic access
- API does not require authentication (internal use)
- Consider rate limiting for production

## Related Documentation

- [LOCAL_DEVELOPMENT.md](LOCAL_DEVELOPMENT.md) - Development setup
- [KAFKA_CONSUMER.md](KAFKA_CONSUMER.md) - Consumer details
- [API.md](API.md) - API reference
- [HAPROXY_GATEWAY.md](HAPROXY_GATEWAY.md) - Production API gateway setup
