/**
 * TidePool Poolstats - Chain detection and API configuration
 * 
 * Reads the chain from the URL path (/pools/bitcoin/ → "bitcoin"),
 * fetches chains.json from the gateway, and configures HTMX API rewrites.
 */

(function() {
    'use strict';

    // Detect chain from URL path: /pools/{chain}/ or /pools/{chain}/page.html
    var pathParts = window.location.pathname.replace(/\/+$/, '').split('/');
    var poolsIdx = pathParts.indexOf('pools');
    var detectedChain = (poolsIdx >= 0 && pathParts.length > poolsIdx + 1) ? pathParts[poolsIdx + 1] : null;

    // Store globally
    window.TIDEPOOL_CHAIN = detectedChain;
    window.TIDEPOOL_CHAIN_CONFIG = null;
    window.TIDEPOOL_CHAINS = [];

    // Build API base from gateway + chain
    function updateApiBase(chain) {
        var gw = window.TIDEPOOL_GATEWAY || '';
        if (gw && chain) {
            window.TIDEPOOL_API_BASE = gw + '/' + chain + '/v1';
        } else {
            window.TIDEPOOL_API_BASE = '/api/v1';
        }
    }

    updateApiBase(detectedChain);

    // Fetch chains.json from gateway (or local /mining/ fallback in dev)
    function loadChains() {
        var gw = window.TIDEPOOL_GATEWAY;
        var chainsUrl = gw ? (gw + '/') : '/mining/';

        fetch(chainsUrl)
            .then(function(r) { return r.json(); })
            .then(function(data) {
                var chains = data.chains || [];
                window.TIDEPOOL_CHAINS = chains;

                // Find config for detected chain (or use default)
                var chainId = detectedChain || data.default || 'bitcoin';
                var config = chains.find(function(c) { return c.id === chainId; });

                if (config) {
                    window.TIDEPOOL_CHAIN = chainId;
                    window.TIDEPOOL_CHAIN_CONFIG = config;
                    updateApiBase(chainId);
                    applyChainConfig(config);
                }

                // Populate chain selector if present
                var selector = document.getElementById('chain-selector');
                if (selector) {
                    selector.innerHTML = '';
                    chains.forEach(function(c) {
                        var opt = document.createElement('option');
                        opt.value = c.id;
                        opt.textContent = c.name;
                        if (c.id === chainId) opt.selected = true;
                        selector.appendChild(opt);
                    });
                    selector.addEventListener('change', function() {
                        // Navigate to the selected chain's pool page
                        var base = window.location.pathname.split('/pools/')[0];
                        window.location.href = base + '/pools/' + this.value + '/';
                    });
                }
            })
            .catch(function(err) {
                console.warn('[poolstats] Failed to load chains.json:', err);
            });
    }

    // Apply chain-specific config to the page
    function applyChainConfig(config) {
        // Set page title
        var titleEl = document.querySelector('[data-chain-name]');
        if (titleEl) titleEl.textContent = config.name || config.id;

        // Stratum URL
        var stratumEl = document.querySelector('[data-stratum-url]');
        if (stratumEl) stratumEl.textContent = config.stratum_url || '';

        // Algorithm
        var algoEl = document.querySelector('[data-algorithm]');
        if (algoEl) algoEl.textContent = config.algorithm || '';

        // Reward
        var rewardEl = document.querySelector('[data-reward-percent]');
        if (rewardEl) rewardEl.textContent = (config.reward_percent || 0) + '%';

        // Block explorer base URL (for building links)
        window.TIDEPOOL_BLOCK_EXPLORER = config.block_explorer_url || '';
        window.TIDEPOOL_TX_EXPLORER = config.tx_explorer_url || '';

        // Apply chain theme color as CSS custom property
        if (config.theme_color) {
            document.documentElement.style.setProperty('--poolstats-accent', config.theme_color);
        }
    }

    // HTMX request rewriting — same pattern as the Monitor frontend
    document.addEventListener('htmx:configRequest', function(event) {
        var path = event.detail.path;
        if (path.indexOf('/api/') === 0) {
            event.detail.path = window.TIDEPOOL_API_BASE + path.substring(4);
        }
    });

    // Load chains on DOM ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', loadChains);
    } else {
        loadChains();
    }

    // Register additional Handlebars helpers for poolstats
    if (typeof Handlebars !== 'undefined') {
        Handlebars.registerHelper('blockExplorerLink', function(hash) {
            var base = window.TIDEPOOL_BLOCK_EXPLORER || '';
            if (!base || !hash) return '#';
            return base + hash;
        });

        Handlebars.registerHelper('txExplorerLink', function(txid) {
            var base = window.TIDEPOOL_TX_EXPLORER || '';
            if (!base || !txid) return '#';
            return base + txid;
        });

        Handlebars.registerHelper('formatNumber', function(num) {
            if (typeof num !== 'number') return num || '0';
            return num.toLocaleString();
        });

        Handlebars.registerHelper('formatDifficulty', function(diff) {
            if (typeof diff !== 'number') return diff || '0';
            if (diff >= 1e15) return (diff / 1e15).toFixed(2) + ' P';
            if (diff >= 1e12) return (diff / 1e12).toFixed(2) + ' T';
            if (diff >= 1e9) return (diff / 1e9).toFixed(2) + ' G';
            if (diff >= 1e6) return (diff / 1e6).toFixed(2) + ' M';
            if (diff >= 1e3) return (diff / 1e3).toFixed(2) + ' K';
            return diff.toFixed(2);
        });

        Handlebars.registerHelper('ifStale', function(stale, options) {
            return stale ? options.fn(this) : options.inverse(this);
        });
    }
})();
